<?php

namespace Drupal\Tests\searchstax\Functional\Update;

use Drupal\FunctionalTests\Update\UpdatePathTestBase;
use Drupal\key\Entity\Key;
use Drupal\search_api\Entity\Server;
use Drupal\searchstax\Plugin\SolrConnector\SearchStaxConnector;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the functionality of the searchstax_update_8104() update hook.
 *
 * @see searchstax_update_8104()
 *
 * @group searchstax
 * @legacy
 */
#[RunTestsInSeparateProcesses]
class SearchStaxUpdate8104Test extends UpdatePathTestBase {

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // We need to manually set the needed entity types as "installed".
    $entity_type_ids = [
      'key',
      'key_config_override',
      'search_api_index',
      'search_api_server',
      'search_api_task',
      'solr_cache',
      'solr_field_type',
      'solr_request_dispatcher',
      'solr_request_handler',
    ];
    foreach ($entity_type_ids as $entity_type_id) {
      $entity_type = \Drupal::getContainer()
        ->get('entity_type.manager')
        ->getDefinition($entity_type_id);
      \Drupal::getContainer()
        ->get('entity_type.listener')
        ->onEntityTypeCreate($entity_type);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function setDatabaseDumpFiles(): void {
    $core_dump_file = glob(DRUPAL_ROOT . '/core/modules/system/tests/fixtures/update/drupal-*.bare.standard.php.gz')[0];
    $this->databaseDumpFiles = [
      $core_dump_file,
      __DIR__ . '/../../../fixtures/update/searchstax-update-8104.php',
    ];
  }

  /**
   * Tests whether searchstax_update_8104() works correctly.
   */
  public function testUpdate8104(): void {
    // Load the original version of all servers to compare them later.
    $servers_before = Server::loadMultiple();

    $this->runUpdates();

    // Verify that the basic configuration has been converted.
    $config = \Drupal::config('searchstax.settings')->get();
    $this->assertEmpty($config['analytics_url'] ?? NULL);
    $this->assertEmpty($config['analytics_key'] ?? NULL);
    $this->assertNotEmpty($config['key_id'] ?? NULL);
    $key = Key::load($config['key_id']);
    $key_data = json_decode($key->getKeyValue(), TRUE, 512, JSON_THROW_ON_ERROR);
    $this->assertEquals('https://app-2.searchstax.com', $key_data['analytics_url']);
    $this->assertEquals('123abc456def', $key_data['analytics_key']);

    $servers_after = Server::loadMultiple();
    // The non-SearchStax Solr server and the Basic Auth server should not have
    // been affected.
    $this->assertEquals(
      $servers_before['other_solr_server']->toArray(),
      $servers_after['other_solr_server']->toArray(),
    );
    $this->assertEquals(
      $servers_before['searchstax_server_basic_auth']->toArray(),
      $servers_after['searchstax_server_basic_auth']->toArray(),
    );

    // The other two should have been affected.
    $get_endpoint_components = new \ReflectionMethod(SearchStaxConnector::class, 'getEndpointComponents');
    $get_endpoint_components->setAccessible(TRUE);
    foreach (['searchstax_server_token_auth', 'searchstax_server_token_auth_2'] as $server_id) {
      $old_server = $servers_before[$server_id]->toArray();
      $new_server = $servers_after[$server_id]->toArray();
      $this->assertNotEquals($old_server, $new_server, "Server $server_id was not affected.");

      $old_config = $old_server['backend_config']['connector_config'];
      $new_config = $new_server['backend_config']['connector_config'];
      $removed_properties = [
        'host',
        'context',
        'core',
        'update_endpoint',
        'update_token',
        'autosuggest_core',
      ];
      foreach ($removed_properties as $property) {
        $this->assertEmpty($new_config['$property'] ?? NULL, "Server $server_id had \"$property\" config still stored.");
      }
      $this->assertNotEmpty($new_config['key_id'] ?? NULL, "Server $server_id did not have \"key_id\" config added.");
      $key = Key::load($new_config['key_id']);
      $key_data = json_decode($key->getKeyValue(), TRUE, 512, JSON_THROW_ON_ERROR);
      $migrated_properties = ['update_endpoint', 'update_token'];
      if (!empty($old_config['autosuggest_core'])) {
        $migrated_properties[] = 'autosuggest_core';
      }
      $this->assertEquals($migrated_properties, array_keys($key_data), "Unexpected properties stored in new key for server $server_id.");
      foreach ($migrated_properties as $property) {
        $this->assertEquals($old_config[$property], $key_data[$property], "Mismatch in \"$property\" for server $server_id.");
      }

      // Verify that SearchStaxConnector::getEndpointComponents() works
      // correctly.
      $connector = $servers_after[$server_id]->getBackend()->getSolrConnector();
      $this->assertEquals(
        [
          'host' => $old_config['host'],
          'context' => $old_config['context'],
          'core' => $old_config['core'],
        ],
        $get_endpoint_components->invoke($connector),
        "Failed to assert correct endpoint data used for server $server_id.",
      );
    }
  }

}
