<?php

namespace Drupal\searchstax\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\Requirement\RequirementSeverity;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Url;
use Drupal\searchstax\Service\SearchStaxServiceInterface;
use Drupal\searchstax\Service\VersionCheckInterface;

/**
 * Provides requirements hook implementations.
 */
class Requirements {

  /**
   * The SearchStax utility service.
   */
  protected SearchStaxServiceInterface $utility;

  /**
   * The SearchStax version check service.
   */
  protected VersionCheckInterface $versionCheck;

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  public function __construct(
    SearchStaxServiceInterface $utility,
    VersionCheckInterface $versionCheck,
    ModuleHandlerInterface $moduleHandler,
    ConfigFactoryInterface $configFactory,
    EntityTypeManagerInterface $entityTypeManager
  ) {
    $this->utility = $utility;
    $this->versionCheck = $versionCheck;
    $this->moduleHandler = $moduleHandler;
    $this->configFactory = $configFactory;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * Implements hook_runtime_requirements().
   */
  #[Hook('runtime_requirements')]
  public function runtimeRequirements(): array {
    $requirements = [];

    if (version_compare(\Drupal::VERSION, '11.2', '>=')) {
      $ok = RequirementSeverity::OK;
      $warning = RequirementSeverity::Warning;
    }
    else {
      $ok = REQUIREMENT_OK;
      $warning = REQUIREMENT_WARNING;
    }

    if ($this->moduleHandler->moduleExists('search_api_searchstax')) {
      $requirements['searchstax_module_conflict'] = [
        'title' => t('SearchStax module conflict'),
        'description' => t('The “Search API SearchStax” module is no longer required for using a SearchStax server with token authentication as long as the “SearchStax” module is installed. You are advised to <a href=":url">uninstall</a> the “Search API SearchStax” module.', [
          ':url' => (new Url('system.modules_uninstall'))->toString(),
        ]),
        'severity' => $warning,
      ];
    }

    if ($this->configFactory->get('searchstax.settings')->get('autosuggest_core')) {
      $requirements['deprecated_autosuggest_core_config'] = [
        'title' => t('Deprecated global SearchStax setting'),
        'description' => t('The global “Auto-suggest core” SearchStax setting has been deprecated. Instead, configure the autosuggest core for each search server separately. Afterwards, go to the <a href=":url">SearchStax settings page</a> and remove the global setting.', [
          ':url' => (new Url('searchstax.admin_settings'))->toString(),
        ]),
        'severity' => $warning,
      ];
    }

    $description = t('The configuration files of all SearchStax apps used on this site are compatible with the currently used version of Drupal.');
    $severity = $ok;
    $searchstax_servers_present = FALSE;
    $server_storage = $this->entityTypeManager->getStorage('search_api_server');
    /** @var \Drupal\search_api\ServerInterface $server */
    foreach ($server_storage->loadMultiple() as $server) {
      if (!$this->utility->isSearchStaxSolrServer($server)) {
        continue;
      }

      // Check the index timeout set for the server.
      $config = $server->getBackendConfig();
      $index_timeout = $config['connector_config']['index_timeout'] ?? 15;
      if ($index_timeout < 15) {
        try {
          $url = $server->toUrl('edit-form')->toString();
        }
        catch (EntityMalformedException $e) {
          // Seems highly unlikely, but catch this nonetheless. We really do not
          // want to throw an exception during requirements check.
          $url = '';
        }
        $requirements['searchstax_index_timeout_' . $server->id()] = [
          'title' => t('Low index timeout'),
          'description' => t('The index timeout for SearchStax server <a href=":url">@server</a> is set to @timeout seconds. It is suggested to increase this to at least 15 seconds to allow indexing with larger batch sizes.',
            [
              '@server' => $server->label() ?: $server->id(),
              ':url' => $url,
              '@timeout' => $index_timeout,
            ]),
          'severity' => $warning,
        ];
      }

      // Check whether the version check has been executed for this server.
      // However, only report a single warning regarding the version check, so
      // skip this if we have already detected one.
      if ($searchstax_servers_present) {
        continue;
      }
      $searchstax_servers_present = TRUE;
      if (!$this->versionCheck->hasCompatibilityDataStored($server)) {
        $description = t('The <a href=":url">SearchStax version check</a> needs to be executed for at least one of your search servers.',
          [
            ':url' => Url::fromRoute('searchstax.version_check')->toString(),
          ]);
        $severity = $warning;
      }
      elseif (!$this->versionCheck->checkCompatibility($server)->isCompatible()) {
        $description = t('The configuration files of at least one SearchStax app used on this site are incompatible with the currently used version of Drupal. Please go to <a href=":url">the SearchStax version check</a> page for details and possible solutions.',
          [
            ':url' => Url::fromRoute('searchstax.version_check')->toString(),
          ]);
        $severity = $warning;
      }
    }
    if ($searchstax_servers_present) {
      $requirements['searchstax_version_check'] = [
        'title' => t('SearchStax app configurations'),
        'description' => $description,
        'severity' => $severity,
      ];
    }

    return $requirements;
  }

}
