<?php

declare(strict_types=1);

namespace Drupal\searchstax\Form;

use Drupal\Core\Config\ConfigBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Utility\Error;
use Drupal\key\KeyRepositoryInterface;
use Drupal\search_api\Display\DisplayPluginManagerInterface;
use Drupal\searchstax\Hook\ServerEditForm;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for changing the module's settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The plugin manager search api display.
   *
   * @var \Drupal\search_api\Display\DisplayPluginManagerInterface
   */
  protected DisplayPluginManagerInterface $displayPluginManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The key repository service, if available.
   */
  protected ?KeyRepositoryInterface $keyRepository = NULL;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    $form = parent::create($container);

    $form->setEntityTypeManager($container->get('entity_type.manager'));
    $form->setDisplayPluginManager($container->get('plugin.manager.search_api.display'));
    $form->setModuleHandler($container->get('module_handler'));
    $form->setLoggerFactory($container->get('logger.factory'));

    // Only inject key repository if the Key module is enabled.
    if ($container->get('module_handler')->moduleExists('key')) {
      $form->setKeyRepository($container->get('key.repository'));
    }

    return $form;
  }

  /**
   * Retrieves the entity type manager.
   *
   * @return \Drupal\Core\Entity\EntityTypeManagerInterface
   *   The entity type manager.
   */
  public function getEntityTypeManager(): EntityTypeManagerInterface {
    return $this->entityTypeManager ?? \Drupal::service('entity_type.manager');
  }

  /**
   * Sets the entity type manager.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The new entity type manager.
   *
   * @return $this
   */
  public function setEntityTypeManager(EntityTypeManagerInterface $entity_type_manager): SettingsForm {
    $this->entityTypeManager = $entity_type_manager;
    return $this;
  }

  /**
   * Retrieves the plugin manager search api display.
   *
   * @return \Drupal\search_api\Display\DisplayPluginManagerInterface
   *   The plugin manager search api display.
   */
  public function getDisplayPluginManager(): DisplayPluginManagerInterface {
    return $this->displayPluginManager ?? \Drupal::service('plugin.manager.search_api.display');
  }

  /**
   * Sets the plugin manager search api display.
   *
   * @param \Drupal\search_api\Display\DisplayPluginManagerInterface $display_plugin_manager
   *   The new plugin manager search api display.
   *
   * @return $this
   */
  public function setDisplayPluginManager(DisplayPluginManagerInterface $display_plugin_manager): self {
    $this->displayPluginManager = $display_plugin_manager;
    return $this;
  }

  /**
   * Retrieves the module handler.
   *
   * @return \Drupal\Core\Extension\ModuleHandlerInterface
   *   The module handler.
   */
  public function getModuleHandler(): ModuleHandlerInterface {
    return $this->moduleHandler ?? \Drupal::service('module_handler');
  }

  /**
   * Sets the module handler.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The new module handler.
   *
   * @return $this
   */
  public function setModuleHandler(ModuleHandlerInterface $module_handler): self {
    $this->moduleHandler = $module_handler;
    return $this;
  }

  /**
   * Retrieves the key repository.
   *
   * @return \Drupal\key\KeyRepositoryInterface|null
   *   The key repository, or NULL if the Key module is not installed.
   */
  public function getKeyRepository(): ?KeyRepositoryInterface {
    if ($this->keyRepository) {
      return $this->keyRepository;
    }
    if ($this->getModuleHandler()->moduleExists('key')) {
      return \Drupal::service('key.repository');
    }
    return NULL;
  }

  /**
   * Sets the key repository.
   *
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository.
   *
   * @return $this
   */
  public function setKeyRepository(KeyRepositoryInterface $key_repository): self {
    $this->keyRepository = $key_repository;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'searchstax_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      'searchstax.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('searchstax.settings');

    $form['js_version'] = [
      '#type' => 'select',
      '#title' => $this->t('Javascript library version'),
      '#description' => $this->t('The version of the SearchStax Studio tracking library to use for your site. You should usually prefer newer versions, but temporarily using an old version might be necessary in case you have custom additions to the tracking code on your site.'),
      '#options' => [
        '2' => '2',
        '3' => '3',
      ],
      '#config_target' => 'searchstax.settings:js_version',
    ];

    // API Credentials configuration section.
    $form['credentials_section'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Analytics Credentials'),
      '#tree' => FALSE,
    ];

    $key_module_exists = $this->moduleHandler->moduleExists('key');
    $key_unused_states = [];
    if ($key_module_exists) {
      $empty_option = $this->t('- Do not use Key module -');
      $form['credentials_section']['key_id'] = [
        '#type' => 'key_select',
        '#title' => $this->t('Analytics Credentials Key'),
        '#empty_option' => $empty_option,
        '#default_value' => $config->get('key_id') ?? '',
        '#description' => $this->t('Select the key that contains your SearchStax Analytics credentials (Analytics URL and Global analytics key) as JSON, or select "@do_not_use" to enter the credentials directly into this form. If a key is used, any credentials entered into this form will be cleared to protect your data. Expected format of the JSON value of the key: {"analytics_url": "https://app.searchstax.com", "analytics_key": "YourSecretKey"}', [
          '@do_not_use' => $empty_option,
        ]),
        '#config_target' => 'searchstax.settings:key_id',
      ];
      $key_unused_states = [
        '#states' => [
          'visible' => [
            ':input[name="key_id"]' => ['value' => ''],
          ],
        ],
      ];
    }

    $form['credentials_section']['analytics_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Analytics URL'),
      '#description' => $this->t("The SearchStudio analytics URL associated with this website. You can find this in your App's settings, under \"Analytics API\". Only available for Javascript library version 3 or higher."),
      '#config_target' => 'searchstax.settings:analytics_url',
      '#states' => [
        'visible' => [
          ':input[name="js_version"]' => [
            'value' => '3',
          ],
        ] + ($key_unused_states['#states']['visible'] ?? []),
      ],
    ];

    $form['credentials_section']['analytics_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Global analytics key'),
      '#description' => $this->t('The SearchStudio analytics key associated with this website. You can override this on a per-search basis below, if you have more than one search app configured for this site.'),
      '#config_target' => 'searchstax.settings:analytics_key',
    ] + $key_unused_states;

    $form['search_specific_analytics_keys'] = [
      '#type' => 'details',
      '#title' => $this->t('Search-specific analytics keys'),
      '#description' => $this->t('If you have multiple search apps configured in SearchStax for this website, please enter the analytics key to use for each individual search below. Leave a field blank to use the global key set above for that search – or disable tracking for a search, in case you have left the global key empty, too.'),
      '#tree' => TRUE,
      '#config_target' => 'searchstax.settings:search_specific_analytics_keys',
    ];
    $displays = $this->getDisplayPluginManager()->getInstances();
    foreach ($displays as $id => $display) {
      $form['search_specific_analytics_keys'][$id] = [
        '#type' => 'textfield',
        '#title' => $display->label(),
        '#default_value' => $config->get("search_specific_analytics_keys.$id"),
      ];
    }

    try {
      $roles = $this->getEntityTypeManager()->getStorage('user_role')
        ->loadMultiple();
      $role_options = [];
      foreach ($roles as $id => $role) {
        $role_options[$id] = $role->label();
      }
      $form['untracked_roles'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Do not track these roles'),
        '#description' => $this->t('Select roles for which you want to disable tracking of search behavior.'),
        '#options' => $role_options,
        '#config_target' => 'searchstax.settings:untracked_roles',
      ];
    }
    catch (\Exception $e) {
      // @todo Remove once we depend on Drupal 10.1+.
      if (method_exists(Error::class, 'logException')) {
        Error::logException($this->logger('searchstax'), $e);
      }
      else {
        /* @noinspection PhpUndefinedFunctionInspection */
        watchdog_exception('searchstax', $e);
      }
    }

    $form['autosuggest_core'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Auto-suggest core'),
      '#description' => $this->t("<strong>This setting has been deprecated.</strong> Instead, set the auto-suggest core to use for each search server separately, in that server's settings. Afterwards, you should remove this setting."),
      '#element_validate' => [[ServerEditForm::class, 'validateAutosuggestCore']],
      '#config_target' => 'searchstax.settings:autosuggest_core',
      // This setting is deprecated. Only display if it has a value, to allow
      // users to remove it.
      '#access' => (bool) $config->get('autosuggest_core'),
    ];

    // @todo Check whether /emselect is available before allowing to enable?
    $form['searches_via_searchstudio'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Re-route searches through SearchStudio'),
      '#description' => $this->t('Re-route all Solr searches through the SearchStudio search handler, to immediately apply adjustments you make in SearchStudio.'),
      '#config_target' => 'searchstax.settings:searches_via_searchstudio',
    ];
    $form['configure_via_searchstudio'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Configure searches via SearchStudio'),
      '#description' => $this->t('If this option is enabled, search settings configured in Drupal will be ignored and the settings configured in SearchStudio will take effect. You can configure which settings exactly will be affected. Fulltext keys, filters and paging parameters are always passed to the Solr server and are never affected by this setting.'),
      '#config_target' => 'searchstax.settings:configure_via_searchstudio',
      '#states' => [
        'visible' => [
          ':input[name="searches_via_searchstudio"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['discard_parameters'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Which Drupal search settings should be ignored?'),
      '#description' => $this->t('The selected Drupal settings will be ignored in the search request sent to SearchStax. Instead, the settings made within SearchStudio will take effect. (Otherwise, the Drupal settings will override the ones in SearchStudio.) This has no effect when there are no Drupal settings for the specific component – for instance, when spellchecking has not been enabled.'),
      '#options' => [
        'keys' => $this->t('Parse mode and searched fields'),
        'highlight' => $this->t('Highlighting settings'),
        'spellcheck' => $this->t('Spellcheck settings'),
        'sort' => $this->t('Sorts'),
        'facets' => $this->t('Facets'),
      ],
      'facets' => [
        '#description' => $this->t('This will in most cases not work correctly with the Facets module, so custom code handling the returned facets will be needed.'),
      ],
      '#config_target' => 'searchstax.settings:discard_parameters',
      '#states' => [
        'visible' => [
          ':input[name="searches_via_searchstudio"]' => ['checked' => TRUE],
          ':input[name="configure_via_searchstudio"]' => ['checked' => TRUE],
        ],
      ],
    ];

    if ($this->manualConfigHandlingNeeded()) {
      static::setConfigDefaultValues($form, $config);
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Sets the default values for the given form element's children.
   *
   * @param array $element
   *   The form element, passed by reference.
   * @param \Drupal\Core\Config\ConfigBase $config
   *   The module config.
   */
  protected static function setConfigDefaultValues(array &$element, ConfigBase $config): void {
    foreach (Element::children($element) as $key) {
      $child = &$element[$key];
      $type = $child['#type'] ?? 'details';
      if ($type === 'fieldset') {
        if (empty($child['#tree'])) {
          static::setConfigDefaultValues($child, $config);
        }
        continue;
      }
      $value = $config->get($key);
      if ($type !== 'details' && $value !== NULL) {
        $child['#default_value'] = $value;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Normalize some of the field values.
    $keys = [
      'discard_parameters',
      'search_specific_analytics_keys',
      'untracked_roles',
    ];
    foreach ($keys as $key) {
      $value = $form_state->getValue($key) ?: [];
      $form_state->setValue($key, array_filter($value));
    }

    parent::validateForm($form, $form_state);

    // Validate Key module configuration, if present.
    if (empty($form['credentials_section']['key_id'])) {
      return;
    }

    $key_id = $form_state->getValue('key_id');
    if (!empty($key_id)) {
      // Validate that the key contains valid JSON with required fields.
      $key = $this->getKeyRepository()->getKey($key_id);
      if ($key) {
        $credentials = json_decode($key->getKeyValue(), TRUE);
        if (json_last_error() !== JSON_ERROR_NONE) {
          $form_state->setError(
            $form['credentials_section']['key_id'],
            $this->t('The selected key does not contain valid JSON.'),
          );
        }
        elseif (!isset($credentials['analytics_url']) || !isset($credentials['analytics_key'])) {
          $form_state->setError(
            $form['credentials_section']['key_id'],
            $this->t('The selected key must contain both "analytics_url" and "analytics_key" fields.'),
          );
        }
      }
      else {
        $form_state->setError(
          $form['credentials_section']['key_id'],
          $this->t('The selected key does not exist anymore.'),
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $was_key_set = $form_state->getValue('key_id', '') !== '';

    if ($this->manualConfigHandlingNeeded()) {
      $form_state->cleanValues();
      $config = $this->configFactory()->getEditable('searchstax.settings');
      $config->setData($form_state->getValues() + $config->get());
      // Handle Key module settings.
      if ($was_key_set) {
        $config
          ->clear('analytics_url')
          ->clear('analytics_key');
      }
      else {
        // Key module not available, save credentials directly.
        $config->clear('key_id');
      }
      $config->save();
    }

    if ($was_key_set) {
      $form_state->setValue('analytics_url', '');
      $form_state->setValue('analytics_key', '');
    }
    else {
      // Key module not available, save credentials directly.
      $form_state->setValue('key_id', '');
    }

    parent::submitForm($form, $form_state);

    // If the "autosuggest_core" field was present but the user cleared it,
    // remove the setting from the config object completely.
    if (
      !empty($form['autosuggest_core']['#access'])
      && $form_state->getValue('autosuggest_core') === ''
    ) {
      $config = $this->configFactory()->getEditable('searchstax.settings');
      $config->clear('autosuggest_core');
      $config->save();
    }
  }

  /**
   * Determines whether this class needs to explicitly handle config changes.
   *
   * Earlier versions of Drupal didn't come with the automatic config handling
   * via the "#config_target" keys in ConfigFormBase, so we need to set default
   * values and save the new config values ourselves.
   *
   * @return bool
   *   TRUE if this form class needs to explicitly handle config changes, FALSE
   *   if this is a Drupal version that includes the automatic handling in
   *   ConfigFormBase.
   *
   * @see \Drupal\Core\Form\ConfigFormBase::copyFormValuesToConfig()
   */
  protected function manualConfigHandlingNeeded(): bool {
    return version_compare(\Drupal::VERSION, '10.2', '<');
  }

}
