<?php

namespace Drupal\searchstax\Hook;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\search_api\ServerInterface;
use Drupal\searchstax\Service\SearchStaxServiceInterface;

/**
 * Provides hook implementations for the SearchStax module.
 */
class ServerEditForm {

  use StringTranslationTrait;

  /**
   * The SearchStax utility service.
   */
  protected SearchStaxServiceInterface $searchstaxUtility;

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  public function __construct(
    SearchStaxServiceInterface $searchstaxUtility,
    ModuleHandlerInterface $module_handler,
    TranslationInterface $string_translation
  ) {
    $this->searchstaxUtility = $searchstaxUtility;
    $this->moduleHandler = $module_handler;
    $this->stringTranslation = $string_translation;
  }

  /**
   * Implements hook_form_FORM_ID_alter() for form "search_api_server_edit_form".
   */
  #[Hook('form_search_api_server_edit_form_alter')]
  public function alterServerEditForm(&$form, FormStateInterface $form_state, string $form_id): void {
    // We need to restrict by form ID here because this function is also called
    // via hook_form_BASE_FORM_ID_alter.
    if (!in_array($form_id, ['search_api_server_form', 'search_api_server_edit_form'])) {
      return;
    }
    $form_builder = $form_state->getFormObject();
    if (!($form_builder instanceof EntityFormInterface)) {
      return;
    }
    $server = $form_builder->getEntity();
    if (
      !($server instanceof ServerInterface)
      || !$this->searchstaxUtility->isSearchStaxSolrServer($server)
      || ($server->getBackendConfig()['connector'] ?? '') === 'searchstax'
      || empty($form['backend_config']['advanced'])
    ) {
      return;
    }

    $element = [
      '#type' => 'textfield',
      '#title' => $this->t('Auto-suggest core'),
      '#description' => $this->t('The “core” part of your SearchStudio auto-suggest URL. This is the second-to-last part of the URL, between “solr” and “emsuggest”: [DOMAIN]/solr/[CORE]/emsuggest. (Only needed if you want to use auto-suggest.)'),
      '#default_value' => $server->getThirdPartySetting('searchstax', 'autosuggest_core'),
      '#element_validate' => [[static::class, 'validateAutosuggestCore']],
      '#parents' => [
        'third_party_settings',
        'searchstax',
        'autosuggest_core',
      ],
    ];
    if (!$this->moduleHandler->moduleExists('search_api_autocomplete')) {
      $suffix = $this->t('Install the <a href=":url">Search API Autocomplete</a> module to use the auto-suggest feature.', [
        ':url' => 'https://www.drupal.org/project/search_api_autocomplete',
      ]);
      $element['#description'] = new FormattableMarkup('@description<br />@suffix', [
        '@description' => $element['#description'],
        '@suffix' => $suffix,
      ]);
    }
    $form['backend_config']['advanced']['searchstax_autosuggest_core'] = $element;
  }

  /**
   * Validates input for the "Auto-suggest core" form field.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public static function validateAutosuggestCore(array &$element, FormStateInterface $form_state): void {
    $value = $form_state->getValue($element['#parents']);
    /* @noinspection PhpStrFunctionsInspection */
    if (strpos($value, '/') !== FALSE) {
      $vars = ['@name' => $element['#title']];
      $message = t('Please enter @name without any slashes (/).', $vars);
      $form_state->setError($element, $message);
    }
  }

}
