<?php

declare(strict_types=1);

namespace Drupal\secret_login\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Returns responses for secret_login routes.
 */
final class SecretLoginController extends ControllerBase {

  /**
   * The current user service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a SecretLoginController object.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user service.
   */
  public function __construct(AccountProxyInterface $current_user) {
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_user')
    );
  }

  /**
   * Generates a login URL for an administrator.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse|\Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response if the user is found and logged in,
   *   or a JSON response with an error message.
   */
  public function generateLoginUrl() {
    $query = \Drupal::entityQuery('user')
      ->condition('roles', 'administrator')
      ->accessCheck(FALSE)
      ->range(0, 1);

    $uids = $query->execute();
    $admin_uid = reset($uids);

    // Load the user entity.
    $user = $this->entityTypeManager()->getStorage('user')->load($admin_uid);
    if ($user) {
      user_login_finalize($user, ['absolute' => TRUE]);
      \Drupal::messenger()->addStatus('Logged in successfully through Secret Login Module.');

      // Redirect to the home page.
      return $this->redirect('<front>');
    }

    return new JsonResponse([
      'status' => 'error',
      'message' => 'Administrator user not found.',
    ], 404);
  }

}
