<?php

declare(strict_types=1);

namespace Drupal\secret_login\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Url;
use Drupal\user\Entity\User;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Controller for Secret Login functionality.
 *
 * @ingroup secret_login
 */
class SecretLoginController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a new SecretLoginController.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database')
    );
  }

  /**
   * Handles direct login with custom URL.
   *
   * @param string $custom_path
   *   The custom path for the login URL.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the user's profile or home page.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   When the URL configuration is not found or inactive.
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   When the user account is not active.
   */
  public function directLogin($custom_path) {
    try {
      $url_config = $this->database->select('secret_login_urls', 'u')
        ->fields('u')
        ->condition('custom_path', $custom_path)
        ->condition('is_active', 1)
        ->execute()
        ->fetchObject();

      if (!$url_config) {
        throw new NotFoundHttpException('Invalid or inactive secret login URL.');
      }

      $user = User::load($url_config->uid);
      if (!$user || !$user->isActive()) {
        throw new AccessDeniedHttpException('User account is not active.');
      }

      user_login_finalize($user);
      $this->messenger()->addStatus($this->t("You're now logged in through the Secret Login URL — A secure and efficient solution crafted to simplify user access."));

      return $this->redirect('entity.user.canonical', ['user' => $user->id()]);
    }
    catch (\Exception $e) {
      $this->messenger()->addError($e->getMessage());
      return $this->redirect('<front>');
    }
  }

  /**
   * Generates a secret login URL for a user.
   *
   * @param string $custom_path
   *   The custom path for the login URL.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing the login URL.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   When the URL configuration is not found or inactive.
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   When the user account is not active.
   */
  public function generateLoginUrl($custom_path) {
    try {
      $url_config = $this->database->select('secret_login_urls', 'u')
        ->fields('u')
        ->condition('custom_path', $custom_path)
        ->condition('is_active', 1)
        ->execute()
        ->fetchObject();

      if (!$url_config) {
        throw new NotFoundHttpException('Invalid or inactive secret login URL.');
      }

      $user = User::load($url_config->uid);
      if (!$user || !$user->isActive()) {
        throw new AccessDeniedHttpException('User account is not active.');
      }

      $token = bin2hex(random_bytes(32));

      $this->database->insert('secret_login_tokens')
        ->fields([
          'uid' => $user->id(),
          'token' => $token,
          'expires' => time() + ($url_config->token_validity * 3600),
          'created' => time(),
        ])
        ->execute();

      $login_url = Url::fromRoute('secret_login.login', ['token' => $token], [
        'absolute' => TRUE,
        'https' => TRUE,
      ])->toString();

      return new JsonResponse([
        'status' => 'success',
        'login_url' => $login_url,
        'expires_in' => $url_config->token_validity * 3600,
      ]);
    }
    catch (\Exception $e) {
      return new JsonResponse([
        'status' => 'error',
        'message' => $e->getMessage(),
      ], 400);
    }
  }

  /**
   * Handles the login process with a token.
   *
   * @param string $token
   *   The login token.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the user's profile or home page.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   When the token is invalid or expired.
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   When the user account is not active.
   */
  public function loginWithToken($token) {
    try {
      $token_data = $this->database->select('secret_login_tokens', 't')
        ->fields('t')
        ->condition('token', $token)
        ->condition('expires', time(), '>')
        ->execute()
        ->fetchObject();

      if (!$token_data) {
        throw new NotFoundHttpException('Invalid or expired login token.');
      }

      $user = User::load($token_data->uid);
      if (!$user || !$user->isActive()) {
        throw new AccessDeniedHttpException('User account is not active.');
      }

      user_login_finalize($user);

      $this->database->delete('secret_login_tokens')
        ->condition('token', $token)
        ->execute();

      $this->messenger()->addStatus($this->t("You're now logged in through the Secret Login URL — A secure and efficient solution crafted to simplify user access."));

      return $this->redirect('entity.user.canonical', ['user' => $user->id()]);
    }
    catch (\Exception $e) {
      $this->messenger()->addError($e->getMessage());
      return $this->redirect('<front>');
    }
  }

}
