<?php

namespace Drupal\secret_login\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Secret Login settings for this site.
 *
 * @ingroup secret_login
 */
class CustomRouteConfigForm extends ConfigFormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The pager manager.
   *
   * @var \Drupal\Core\Pager\PagerManagerInterface
   */
  protected $pagerManager;

  /**
   * The tempstore factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * Number of users to display per page.
   *
   * @var int
   */
  public const USERS_PER_PAGE = 10;

  /**
   * Constructs a new CustomRouteConfigForm.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Pager\PagerManagerInterface $pager_manager
   *   The pager manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The tempstore factory.
   */
  public function __construct(
    Connection $database,
    MessengerInterface $messenger,
    LoggerChannelFactoryInterface $logger_factory,
    EntityTypeManagerInterface $entity_type_manager,
    PagerManagerInterface $pager_manager,
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    PrivateTempStoreFactory $temp_store_factory,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->database = $database;
    $this->messenger = $messenger;
    $this->loggerFactory = $logger_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->pagerManager = $pager_manager;
    $this->tempStoreFactory = $temp_store_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('messenger'),
      $container->get('logger.factory'),
      $container->get('entity_type.manager'),
      $container->get('pager.manager'),
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('tempstore.private'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['secret_login.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'secret_login_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);

    // Add search container.
    $form['search_container'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['user-search-container']],
    ];

    // Get current request.
    $request = \Drupal::request();

    // Get search term from query parameter, form state, or tempstore.
    $search_term = $request->query->get('search')
      ?? $form_state->getValue('search')
      ?? $this->tempStoreFactory->get('secret_login')->get('search_term')
      ?? '';
    $form_state->set('search_term', $search_term);

    // Add search field.
    $form['search_container']['search'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search Users'),
      '#description' => $this->t('Enter username or email to filter users'),
      '#size' => 30,
      '#default_value' => $search_term,
      '#attributes' => ['class' => ['user-search-field']],
      '#prefix' => '<div class="search-field-wrapper">',
      '#suffix' => '</div>',
    ];

    // Add search and reset buttons container.
    $form['search_container']['buttons'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['search-buttons-wrapper']],
    ];

    // Add search button.
    $form['search_container']['buttons']['search_button'] = [
      '#type' => 'submit',
      '#value' => $this->t('Search'),
      '#name' => 'search_users',
      '#attributes' => ['class' => ['search-button']],
      '#submit' => ['::searchUsers'],
    ];

    // Add reset button.
    $form['search_container']['buttons']['reset_button'] = [
      '#type' => 'submit',
      '#value' => $this->t('Reset Search'),
      '#name' => 'reset_search',
      '#attributes' => ['class' => ['reset-button']],
      '#submit' => ['::resetSearch'],
    ];

    // Get all active users except anonymous.
    $query = $this->entityTypeManager->getStorage('user')->getQuery()
      ->condition('status', 1)
      ->condition('uid', 0, '>');

    // Add search filter if search term is present.
    $tempstore = $this->tempStoreFactory->get('secret_login');
    $search_term = $form_state->getValue('search') ?? $tempstore->get('search_term') ?? '';
    if (!empty($search_term)) {
      $search_term = '%' . $this->database->escapeLike($search_term) . '%';
      $or_group = $query->orConditionGroup()
        ->condition('name', $search_term, 'LIKE')
        ->condition('mail', $search_term, 'LIKE');
      $query->condition($or_group);
    }

    $query->sort('name', 'ASC')
      ->accessCheck(FALSE);

    // Get total count for pager.
    $count_query = clone $query;
    $total = $count_query->count()->execute();

    // Add pager.
    $this->pagerManager->createPager($total, self::USERS_PER_PAGE);
    $page = $this->pagerManager->findPage();
    $start = $page * self::USERS_PER_PAGE;

    // Add range to query.
    $query->range($start, self::USERS_PER_PAGE);
    $uids = $query->execute();
    $users = $this->entityTypeManager->getStorage('user')->loadMultiple($uids);

    // Get existing URLs.
    $existing_urls = $this->database->select('secret_login_urls', 'u')
      ->fields('u')
      ->execute()
      ->fetchAllAssoc('uid');

    $form['users'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('User Secret Login URLs'),
      '#description' => $this->t('Configure secret login URLs for each user.'),
      '#collapsible' => FALSE,
    ];

    foreach ($users as $user) {
      $form['users']['user_' . $user->id()] = [
        '#type' => 'fieldset',
        '#title' => $this->t('@name (@email)', [
          '@name' => $user->getAccountName(),
          '@email' => $user->getEmail(),
        ]),
        '#collapsible' => TRUE,
        '#collapsed' => TRUE,
      ];

      $custom_path = $existing_urls[$user->id()]->custom_path ?? '';
      $base_url = \Drupal::request()->getSchemeAndHttpHost();
      $preview_path = $custom_path ?: 'your-path';

      $form['users']['user_' . $user->id()]['custom_path_' . $user->id()] = [
        '#type' => 'textfield',
        '#title' => $this->t('Custom Login URL Path'),
        '#description' => $this->t('Enter a custom path for this user\'s login URL. Example: "my-secret-login" or "special-access". The full URL will be: @url', [
          '@url' => $base_url . '/secret-login/access/' . $preview_path,
        ]),
        '#default_value' => $custom_path,
        '#required' => FALSE,
        '#pattern' => '[a-zA-Z0-9-]+',
        '#pattern_error' => $this->t('Only letters, numbers, and hyphens are allowed.'),
      ];

      if (!empty($custom_path)) {

        // Initialize token variables.
        $token = NULL;
        $expires = NULL;
        // Get token validity from existing configuration.
        $token_validity = $existing_urls[$user->id()]->token_validity ?? 1;

        // Check if token exists and is expired.
        $existing_token = $this->database->select('secret_login_tokens', 't')
          ->fields('t')
          ->condition('uid', $user->id())
          ->orderBy('expires', 'DESC')
          ->range(0, 1)
          ->execute()
          ->fetchObject();

        $is_expired = FALSE;
        if ($existing_token) {
          // Check if token is expired based on time.
          $current_time = time();
          $is_expired = $current_time >= $existing_token->expires;

          // If token is expired, delete it.
          if ($is_expired) {
            $this->database->delete('secret_login_tokens')
              ->condition('uid', $user->id())
              ->execute();
            $token = NULL;
          }
          else {
            $token = $existing_token->token;
            $expires = $existing_token->expires;
          }
        }

        // Only generate a new token if there isn't a valid one.
        if (empty($token)) {
          // Generate a new token.
          $token = bin2hex(random_bytes(32));
          $current_time = time();
          $expires = $current_time + ($token_validity * 3600);

          // Store the new token.
          $this->database->insert('secret_login_tokens')
            ->fields([
              'uid' => $user->id(),
              'token' => $token,
              'expires' => $expires,
              'created' => $current_time,
            ])
            ->execute();
        }

        // Generate the login URL.
        $login_url = \Drupal::request()->getSchemeAndHttpHost() . '/secret-login/' . $token;

        // Add CSS for highlighting.
        $form['#attached']['html_head'][] = [
          [
            '#type' => 'html_tag',
            '#tag' => 'style',
            '#value' => '
              .token-url {
                background-color: #f5f5f5;
                padding: 10px;
                border: 1px solid #ddd;
                border-radius: 4px;
                margin: 5px 0;
                font-family: monospace;
                word-break: break-all;
                display: inline-block;
                min-width: 300px;
              }
              .token-url.valid {
                border-color: #4CAF50;
                background-color: #E8F5E9;
              }
              .token-url.expired {
                border-color: #F44336;
                background-color: #FFEBEE;
              }
              .token-url.disabled {
                border-color: #9E9E9E;
                background-color: #F5F5F5;
                opacity: 0.7;
              }
              .token-status {
                margin-top: 5px;
                font-size: 0.9em;
              }
              .token-status.valid {
                color: #4CAF50;
              }
              .token-status.expired {
                color: #F44336;
              }
              .token-status.disabled {
                color: #9E9E9E;
              }
            ',
          ],
          'token-url-styles',
        ];

        // Check if token is expired and if login URL is enabled.
        $is_enabled = $existing_urls[$user->id()]->is_active ?? 1;

        if (!$is_enabled) {
          $token_class = 'disabled';
          $status_text = $this->t('Login URL is disabled. Enable it to use this URL.');
        }
        else {
          $token_class = $is_expired ? 'expired' : 'valid';
          if ($is_expired) {
            $status_text = $this->t('Token has expired. Please save the configuration to generate a new token.');
          }
          else {
            $remaining_time = ceil(($expires - time()) / 60);
            $status_text = $this->t('Token is valid for @minutes minutes.', ['@minutes' => $remaining_time]);
          }
        }

        $form['users']['user_' . $user->id()]['login_url'] = [
          '#type' => 'item',
          '#title' => $this->t('One-Time Login URL Token'),
          '#description' => $this->t('Use this URL to log in as this user.'),
          '#markup' => '<div class="token-url ' . $token_class . '">' . $login_url . '</div><div class="token-status ' . $token_class . '">' . $status_text . '</div>',
        ];
      }

      $form['users']['user_' . $user->id()]['token_validity_' . $user->id()] = [
        '#type' => 'number',
        '#title' => $this->t('Token Validity Period (hours)'),
        '#value' => 1,
        '#min' => 0.1,
        '#step' => 0.1,
        '#required' => TRUE,
        '#disabled' => TRUE,
      ];

      $form['users']['user_' . $user->id()]['is_active_' . $user->id()] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable Login URL'),
        '#description' => $this->t('Check this to enable the secret login URL for this user.'),
        '#default_value' => $existing_urls[$user->id()]->is_active ?? 1,
      ];
    }

    // Add pager element with search parameter.
    $form['pager'] = [
      '#type' => 'pager',
      '#quantity' => 5,
      '#parameters' => ['search' => $search_term],
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save configuration'),
      '#button_type' => 'primary',
    ];

    $form['actions']['reset'] = [
      '#type' => 'submit',
      '#value' => $this->t('Reset to defaults'),
      '#submit' => ['::resetForm'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $transaction = $this->database->startTransaction();
    try {
      // Delete all existing URLs.
      $this->database->delete('secret_login_urls')->execute();

      // Insert new URLs.
      $query = $this->database->insert('secret_login_urls')
        ->fields(['uid', 'custom_path', 'token_validity', 'is_active']);

      foreach ($form_state->getValues() as $key => $value) {
        if (strpos($key, 'custom_path_') === 0) {
          $uid = substr($key, strlen('custom_path_'));
          if (!empty($value)) {
            $query->values([
              'uid' => $uid,
              'custom_path' => $value,
              'token_validity' => $form_state->getValue('token_validity_' . $uid),
              'is_active' => $form_state->getValue('is_active_' . $uid),
            ]);
          }
        }
      }

      $query->execute();
      $this->messenger->addStatus($this->t('The configuration options have been saved.'));
    }
    catch (\Exception $e) {
      $transaction->rollBack();
      $this->messenger->addError($this->t('An error occurred while saving the configuration.'));
      $this->loggerFactory->get('secret_login')->error($e->getMessage());
    }

    parent::submitForm($form, $form_state);
  }

  /**
   * Resets the form to default values.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function resetForm(array &$form, FormStateInterface $form_state) {
    // Start a transaction for atomic operations.
    $transaction = $this->database->startTransaction();
    try {
      // Delete all existing URLs.
      $this->database->delete('secret_login_urls')->execute();
      // Delete all existing tokens.
      $this->database->delete('secret_login_tokens')->execute();
      $this->messenger->addStatus($this->t('The configuration options have been reset to their default values.'));
    }
    catch (\Exception $e) {
      $transaction->rollBack();
      $this->messenger->addError($this->t('An error occurred while resetting the configuration.'));
      $this->loggerFactory->get('secret_login')->error($e->getMessage());
    }
    $form_state->setRedirect('secret_login.settings_form');
  }

  /**
   * Submit handler for the search button.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function searchUsers(array &$form, FormStateInterface $form_state) {
    // Store search term in tempstore.
    $search_term = $form_state->getValue('search');
    $tempstore = $this->tempStoreFactory->get('secret_login');
    $tempstore->set('search_term', $search_term);

    // Redirect to same page with search parameter.
    $form_state->setRedirect('<current>', [], [
      'query' => ['search' => $search_term],
    ]);
  }

  /**
   * Submit handler for the reset search button.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function resetSearch(array &$form, FormStateInterface $form_state) {
    // Clear the search field and tempstore.
    $form_state->setValue('search', '');
    $tempstore = $this->tempStoreFactory->get('secret_login');
    $tempstore->delete('search_term');

    // Redirect to same page without search parameter.
    $form_state->setRedirect('<current>');
  }

}
