<?php

namespace Drupal\sector_workflow\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\content_moderation\Form\EntityModerationForm;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'ContentModerationBlock' block.
 *
 * This block displays the content moderation form for nodes that have
 * moderation states enabled.
 *
 * @Block(
 *   id = "content_moderation_block",
 *   admin_label = @Translation("Content moderation block"),
 *   category = @Translation("Content Moderation"),
 * )
 */
class ContentModerationBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * Constructs a new ContentModerationBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    ModuleHandlerInterface $module_handler,
    FormBuilderInterface $form_builder,
    RouteMatchInterface $route_match
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->moduleHandler = $module_handler;
    $this->formBuilder = $form_builder;
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('module_handler'),
      $container->get('form_builder'),
      $container->get('current_route_match')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $build = [];

    /** @var \Drupal\node\NodeInterface|null $node */
    $node = $this->routeMatch->getParameter('node');

    if (!$this->shouldRenderModerationForm($node)) {
      return $build;
    }

    // Get the content moderation form render array.
    // @phpstan-ignore-next-line
    $content_moderation_form = $this->formBuilder->getForm(EntityModerationForm::class, $node);

    if (!empty($content_moderation_form)) {
      $build['content_moderation_block'] = $content_moderation_form;
    }

    return $build;
  }

  /**
   * Determines if the moderation form should be rendered for the given node.
   *
   * @param \Drupal\node\NodeInterface|null $node
   *   The node to check, or NULL if no node is available.
   *
   * @return bool
   *   TRUE if the moderation form should be rendered, FALSE otherwise.
   */
  protected function shouldRenderModerationForm(?NodeInterface $node): bool {
    if (!$node) {
      return FALSE;
    }

    if (!$node->hasField('moderation_state') || $node->get('moderation_state')->isEmpty()) {
      return FALSE;
    }

    // Only render the form for the latest revision, default revision, or when
    // the revision ID matches the node ID (current revision).
    return $node->isLatestRevision() ||
           $node->isDefaultRevision() ||
           $node->getRevisionId() == $node->id();
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    // This block content is dynamic and depends on the current node's
    // moderation state, so it cannot be cached.
    return 0;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    // Add cache contexts to ensure the block varies appropriately.
    return Cache::mergeContexts(parent::getCacheContexts(), [
      'route',
      'user.permissions',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    // Add cache tags for the node so the block is invalidated when the node
    // is updated.
    $cache_tags = parent::getCacheTags();

    $node = $this->routeMatch->getParameter('node');
    if ($node instanceof NodeInterface) {
      $cache_tags = Cache::mergeTags($cache_tags, $node->getCacheTags());
    }

    return $cache_tags;
  }

}
