<?php

namespace Drupal\secure_password_reset_log\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form to clear all password reset logs.
 */
class ClearLogsForm extends ConfirmFormBase {

  /**
   * Database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * Constructs the ClearLogsForm.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'secure_password_reset_log_clear_all_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): string {
    return $this->t('Are you sure you want to clear all password reset logs? This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return Url::fromRoute('secure_password_reset_log.logs');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): string {
    return $this->t('Clear all');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {

    // Remove all password reset logs.
    $this->database->truncate('secure_password_reset_log')->execute();

    // Remove related flood control locks.
    $this->database->delete('flood')
      ->condition('event', 'password_reset_attempt')
      ->execute();

    $this->messenger()->addStatus(
      $this->t('All password reset logs and flood restrictions have been cleared.')
    );

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
