<?php

namespace Drupal\secure_password_reset_log\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for password reset flood protection.
 */
class FloodSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['secure_password_reset_log.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'secure_password_reset_log_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('secure_password_reset_log.settings');

    $form['flood_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum password reset attempts'),
      '#description' => $this->t('Number of allowed attempts before blocking the IP.'),
      '#default_value' => $config->get('flood_limit') ?? 5,
      '#min' => 1,
      '#required' => TRUE,
    ];

    $form['flood_window'] = [
      '#type' => 'number',
      '#title' => $this->t('Time window (seconds)'),
      '#description' => $this->t('Time period in seconds for attempt calculation. Example: 3600 = 1 hour.'),
      '#default_value' => $config->get('flood_window') ?? 3600,
      '#min' => 60,
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    if ($form_state->getValue('flood_limit') <= 0) {
      $form_state->setErrorByName('flood_limit', $this->t('Maximum attempts must be greater than zero.'));
    }

    if ($form_state->getValue('flood_window') < 60) {
      $form_state->setErrorByName('flood_window', $this->t('Time window must be at least 60 seconds.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('secure_password_reset_log.settings')
      ->set('flood_limit', $form_state->getValue('flood_limit'))
      ->set('flood_window', $form_state->getValue('flood_window'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
