<?php

namespace Drupal\secure_password_reset_log\Plugin\views\field;

use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides Delete link for log entries.
 *
 * @ViewsField("secure_password_reset_log_delete_link")
 */
class DeleteLogLink extends FieldPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The current user account service.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * Constructs a new DeleteLogLink object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user account service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, AccountInterface $current_user) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user')
    );
  }

  /**
   * We don't need to add anything to the query for this field.
   */
  public function query() {
    // No database changes needed.
  }

  /**
   * Renders the delete link.
   */
  public function render(ResultRow $values) {
    // We assume 'lid' is available in the view results.
    $lid = $values->lid ?? NULL;

    if (!$lid) {
      return '';
    }

    // Check permissions before rendering the link.
    if (!$this->currentUser->hasPermission('administer secure password reset logs')) {
      return '';
    }

    $url = Url::fromRoute('secure_password_reset_log.delete', ['log' => $lid]);

    // Return a renderable array for the link.
    return Link::fromTextAndUrl($this->t('Delete'), $url)->toRenderable();
  }

}
