<?php

namespace Drupal\secure_password_reset_log\Controller;

use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Flood\FloodInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\PageCache\ResponsePolicy\KillSwitch;
use Drupal\Component\Utility\Html;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Messenger\MessengerTrait;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Controller for displaying and managing secure password reset logs.
 */
class LogController implements ContainerInjectionInterface {
  use AutowireTrait;
  use StringTranslationTrait;
  use MessengerTrait;
  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The flood control service.
   *
   * @var \Drupal\Core\Flood\FloodInterface
   */
  protected $flood;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The page cache kill switch.
   *
   * @var \Drupal\Core\PageCache\ResponsePolicy\KillSwitch
   */
  protected $killSwitch;

  /**
   * The current user service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a LogController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Flood\FloodInterface $flood
   *   The flood control service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\PageCache\ResponsePolicy\KillSwitch $killSwitch
   *   The page cache kill switch.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user service.
   */
  public function __construct(
    Connection $database,
    FloodInterface $flood,
    DateFormatterInterface $date_formatter,
    KillSwitch $killSwitch,
    AccountProxyInterface $currentUser,
  ) {
    $this->database = $database;
    $this->flood = $flood;
    $this->dateFormatter = $date_formatter;
    $this->killSwitch = $killSwitch;
    $this->currentUser = $currentUser;
  }

  /**
   * Displays the log table.
   *
   * @return array
   *   A render array for the log table page.
   */
  public function view(): array {
    $header = [
      'lid' => $this->t('Log ID'),
      'action' => $this->t('Action'),
      'timestamp' => $this->t('Time'),
      'hostname' => $this->t('Hostname/IP'),
      'details' => $this->t('Details'),
      'mail' => $this->t('Email'),
      'uid' => $this->t('User ID'),
      'status' => $this->t('User Status'),
      'operations' => $this->t('Operations'),
    ];

    $limit = 50;

    $results = $this->database->select('secure_password_reset_log', 'l')
      ->fields('l')
      ->orderBy('timestamp', 'DESC')
      ->range(0, $limit)
      ->execute()
      ->fetchAll();

    $uids = array_unique(array_filter(array_column($results, 'uid')));

    $users = [];
    if ($uids) {
      $users = $this->database->select('users_field_data', 'u')
        ->fields('u', ['uid', 'mail'])
        ->condition('uid', $uids, 'IN')
        ->execute()
        ->fetchAllKeyed();
    }

    $rows = [];

    foreach ($results as $record) {
      $operations = [];

      if ($this->currentUser->hasPermission('administer secure password reset logs')) {
        $operations[] = Link::fromTextAndUrl(
              $this->t('Delete'),
              Url::fromRoute('secure_password_reset_log.delete', ['log' => $record->lid])
          )->toRenderable();
      }

      $status = ($record->user_exists ?? 1)
            ? '<span class="user-valid">✅ Exists</span>'
            : '<span class="user-invalid">❌ Not Found</span>';

      $rows[] = [
        'lid' => $record->lid,
        'action' => $record->action,
        'timestamp' => $this->dateFormatter->format($record->timestamp, 'short'),
        'hostname' => $record->hostname,
        'details' => Html::escape($record->details),
        'mail' => $users[$record->uid] ?? $record->email,
        'uid' => $record->uid ?: 0,
        'status' => [
          'data' => [
            '#markup' => $status,
          ],
        ],
        'operations' => [
          'data' => [
            '#theme' => 'links',
            '#links' => array_map(fn($op) => [
              'title' => $op['#title'],
              'url' => $op['#url'],
            ], $operations),
          ],
        ],
      ];
    }

    $build['clear_all'] = [
      '#type' => 'link',
      '#title' => $this->t('Clear all logs & reset flood'),
      '#url' => Url::fromRoute('secure_password_reset_log.clear_all'),
      '#attributes' => ['class' => ['button', 'button--danger']],
      '#access' => $this->currentUser->hasPermission('administer secure password reset logs'),
    ];

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No password reset logs found.'),
    ];

    return $build;
  }

  /**
   * Deletes a specific log entry and clears associated flood entries.
   *
   * @param int $log
   *   The ID of the log entry to delete.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the log view page.
   */
  public function delete($log) {
    $record = $this->database->select('secure_password_reset_log', 'l')
      ->fields('l', ['hostname'])
      ->condition('lid', $log)
      ->execute()
      ->fetchAssoc();

    if ($record) {
      $this->flood->clear('password_reset_attempt', $record['hostname']);
      $this->killSwitch->trigger();

      $this->database->delete('secure_password_reset_log')
        ->condition('lid', $log)
        ->execute();
    }

    $this->messenger()->addStatus($this->t('Log and flood entry cleared.'));
    return new RedirectResponse(Url::fromRoute('secure_password_reset_log.view')->toString());
  }

  /**
   * Clears all log entries and all associated flood records.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the log view page.
   */
  public function clearAll() {
    $ips = $this->database->select('secure_password_reset_log', 'l')
      ->fields('l', ['hostname'])
      ->execute()
      ->fetchCol();

    foreach ($ips as $ip) {
      $this->flood->clear('password_reset_attempt', $ip);
    }

    $this->killSwitch->trigger();
    $this->database->truncate('secure_password_reset_log')->execute();

    $this->messenger()->addStatus($this->t('All logs and flood records cleared.'));
    return new RedirectResponse(Url::fromRoute('secure_password_reset_log.view')->toString());
  }

}
