<?php

namespace Drupal\secure_password_reset_log\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Flood\FloodInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Component\Datetime\TimeInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Handles secure logging and flood detection for password reset attempts.
 */
class PasswordResetSubscriber implements EventSubscriberInterface {
  use StringTranslationTrait;

  /**
   * Maximum allowed password reset attempts before lockout.
   *
   * @see \Drupal\secure_password_reset_log\EventSubscriber\SecurePasswordResetEventSubscriber
   */
  public const DEFAULT_MAX_ATTEMPTS = 5;

  /**
   * Time window (in seconds) for counting reset attempts.
   *
   * @see \Drupal\secure_password_reset_log\EventSubscriber\SecurePasswordResetEventSubscriber
   */
  const DEFAULT_WINDOW = 3600;

  /**
   * The current user service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The flood control service.
   *
   * @var \Drupal\Core\Flood\FloodInterface
   */
  protected $flood;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a new PasswordResetSubscriber object.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Flood\FloodInterface $flood
   *   The flood control service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   */
  public function __construct(
    AccountProxyInterface $current_user,
    Connection $database,
    FloodInterface $flood,
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entityTypeManager,
    TimeInterface $time,
    TranslationInterface $string_translation,
  ) {
    $this->currentUser = $current_user;
    $this->database = $database;
    $this->flood = $flood;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entityTypeManager;
    $this->time = $time;
    $this->setStringTranslation($string_translation);
  }

  /**
   * Handles the request event to log and flood control password resets.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The event object.
   */
  public function onRequest(RequestEvent $event): void {
    if (!$event->isMainRequest()) {
      return;
    }

    $request = $event->getRequest();

    if ($request->attributes->get('_route') !== 'user.pass') {
      return;
    }

    // Only log when form is submitted, not when the page is viewed.
    if (!$request->isMethod('POST')) {
      return;
    }

    $ip = $request->getClientIp();
    $config = $this->configFactory->get('secure_password_reset_log.settings');

    // Use defined constants as fallbacks.
    $limit = $config->get('flood_limit') ?? self::DEFAULT_MAX_ATTEMPTS;
    $window = $config->get('flood_window') ?? self::DEFAULT_WINDOW;

    // Extract submitted email/username.
    $email = $request->request->get('name');
    $uid = 0;

    if ($email) {
      // In a real scenario, you should inject the entityTypeManager.
      // Since this is an event subscriber, we must use dependency injection.
      // If you added EntityTypeManagerInterface to the constructor,
      // you'd use $this->entityTypeManager.
      // For a quick fix without changing the constructor:
      $users = $this->entityTypeManager
        ->getStorage('user')
        ->loadByProperties(['mail' => $email]);

      if ($users) {
        $user = reset($users);
        $uid = $user->id();
      }
    }

    // Flood check.
    if (!$this->flood->isAllowed('password_reset_attempt', $limit, $window, $ip)) {
      $this->logEvent(
        $uid,
        'flood_blocked',
        $uid > 0 ? 'Password reset blocked due to excessive attempts.'
        : 'Password reset blocked due to excessive attempts for non-existing email.',
        $ip,
        $email
      );

      $event->setResponse(new Response(
        $this->t('Too many password reset attempts. Please try again later.'),
        429
      ));
      return;
    }

    // Register attempt only after confirmed valid.
    $this->flood->register('password_reset_attempt', $window, $ip);

    $this->logEvent(
      $uid,
      'reset_request_attempt',
      ($uid > 0)
        ? 'Password reset attempted.'
        : 'Password reset attempted for non-existing email.',
      $ip,
      $email
    );
  }

  /**
   * Logs a password reset related event to the custom database table.
   *
   * @param int $uid
   *   The user ID associated with the event (0 if anonymous or not found).
   * @param string $action
   *   The action type (e.g., 'flood_blocked', 'reset_request_attempt').
   * @param string $details
   *   Additional details about the event.
   * @param string $ip
   *   The hostname or IP address of the requester.
   * @param string $email
   *   The email address used in the request.
   */
  protected function logEvent(int $uid, string $action, string $details, string $ip, string $email = ''): void {
    $this->database->insert('secure_password_reset_log')
      ->fields([
        'uid' => $uid,
        'email' => $email,
        'user_exists' => ($uid > 0) ? 1 : 0,
        'action' => $action,
        'timestamp' => $this->time->getRequestTime(),
        'hostname' => $ip,
        'details' => $details,
      ])
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::REQUEST => ['onRequest', 30],
    ];
  }

}
