<?php

namespace Drupal\secure_password_reset_log\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form to delete a single password reset log entry.
 */
class DeleteLogForm extends ConfirmFormBase {

  /**
   * The ID of the log to delete.
   *
   * @var int
   */
  protected int $logId;

  /**
   * Database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs the DeleteLogForm.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'secure_password_reset_log_delete_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): string {
    return $this->t('Are you sure you want to delete this password reset log entry?');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return Url::fromRoute('secure_password_reset_log.logs');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): string {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $log = NULL): array {
    $this->logId = (int) $log;
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {

    if ($this->logId) {

      // Delete the specific log entry.
      $this->database->delete('secure_password_reset_log')
        ->condition('lid', $this->logId)
        ->execute();

      // Clear flood lockouts for password reset attempts.
      $this->database->delete('flood')
        ->condition('event', 'password_reset_attempt')
        ->execute();

      $this->messenger()->addStatus(
        $this->t('Log entry and associated flood records have been cleared.')
      );
    }

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
