<?php

namespace Drupal\security_login_secure\Form\Section;

use Drupal\security_login_secure\Form\WebsiteSecurityConfiguration;
use Drupal\Core\Form\FormStateInterface;
use Drupal\security_login_secure\Utilities;
use Drupal\Core\Render\Markup;

/**
 * Form handler for IP-based brute force protection settings.
 */
class BruteForceIpForm extends WebsiteSecurityConfiguration
{

    /**
     * {@inheritdoc}
     */
    public function buildBruteForceIpForm(array &$form, FormStateInterface $form_state)
    {
        $info_icon_url = Utilities::get_info_icon();
        $db_var = $this->config('security_login_secure.settings');
        global $base_url;
        $form['bruteforce_ip'] = [
            '#type' => 'details',
            '#title' => 'IP Based Brute Force Protection',
            '#group' => 'information',
        ];

        $form['bruteforce_ip']['description'] = [
            '#type' => 'item',
            '#markup' => 'This feature protects your site from attacks by blocking IP which tries to login with random usernames and passwords.<hr>',
        ];

        $form['bruteforce_ip']['website_security_ip_enable_bruteforce'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Enable IP Based Brute Force Protection</b>'),
            '#default_value' => $db_var->get('website_security_ip_enable_bruteforce')
        ];

        $prefixname = '<div class="ns_row"><div class="ns_name">';
        $suffixname = '</div>';
        $prefixvalue = '<div class="ns_value">';
        $suffixvalue = '</div></div>';

        $form['bruteforce_ip']['set_of_limit_login_features'] = [
            '#type' => 'fieldset',
            '#attributes' => ['style' => ['padding:14px 12px 14px;color:#34495e;border-radius:4px;']],
            '#states' => [
                'visible' => [
                    ':input[name="website_security_ip_enable_bruteforce"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $tooltip_markup = $this->t(
            'Track time to check for security violations (hours) 
			<div class="ns_tooltip">
				<img src=":url" alt="info icon" height="20px" width="15px">
			</div>
			<div class="ns_tooltiptext">
				The time in hours for which the failed login attempts are monitored. After that time, the attempts are deleted and will never be considered again.<br> 
				<b>Note:</b> Provide 0 if you do not want to enable the feature.
			</div>',
            [':url' => $info_icon_url]
        );

        $form['bruteforce_ip']['set_of_limit_login_features']['website_security_ip_track_time'] = [
            '#markup' => Markup::create($tooltip_markup),
            '#prefix' => $prefixname,
            '#suffix' => $suffixname,
        ];

        $form['bruteforce_ip']['set_of_limit_login_features']['website_security_ip_track_time_value'] = [
            '#type' => 'number',
            '#min' => 0,
            '#prefix' => $prefixvalue,
            '#suffix' => $suffixvalue,
            '#default_value' => $db_var->get('website_security_ip_track_time')
        ];

        $tooltip_markup = $this->t('Number of login failures before blocking an IP <div class="ns_tooltip"><img src=":info_icon_url" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">The number of failed login attempts through an IP before it gets blocked. After that count, the IP is blocked, and user can never login using that IP until it is unblocked by admin or after the time provided below.</div>', [':info_icon_url' => $info_icon_url]);

        $form['bruteforce_ip']['set_of_limit_login_features']['website_security_ip_allowed_attempts_name'] = [
            '#markup' => Markup::create($tooltip_markup),
            '#prefix' => $prefixname,
            '#suffix' => $suffixname,
        ];

        $form['bruteforce_ip']['set_of_limit_login_features']['website_security_ip_allowed_attempts_value'] = [
            '#type' => 'number',
            '#min' => 1,
            '#prefix' => $prefixvalue,
            '#suffix' => $suffixvalue,
            '#default_value' => $db_var->get('website_security_ip_allowed_attempts')
        ];

        $tooltip_markup = $this->t('Time period for which IP should be blocked (hours) <div class="ns_tooltip"><img src=":info_icon_url" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">The time in hours for which the IP will remain in blocked state. After that time, the IP will be unblocked.<br> <b>Note: </b>Provide 0 if you want to permanently block an IP.</div>', [':info_icon_url' => $info_icon_url]);

        $form['bruteforce_ip']['set_of_limit_login_features']['website_security_ip_blocked'] = [
            '#markup' => Markup::create($tooltip_markup),
            '#prefix' => $prefixname,
            '#suffix' => $suffixname,
        ];

        $form['bruteforce_ip']['set_of_limit_login_features']['website_security_ip_blocked_time'] = [
            '#type' => 'number',
            '#min' => 0,
            '#prefix' => $prefixvalue,
            '#suffix' => $suffixvalue,
            '#default_value' => $db_var->get('website_security_ip_block_time_period')
        ];

        $tooltip_markup = $this->t('Number of login failures before detecting an attack <div class="ns_tooltip"><img src=":info_icon_url" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">The number of failed login attempts through an IP that can be considered as an attack. After that count, the admin gets a notification email about the attack.<br> <b>Note: </b>Provide a number less than the allowed attempts or else provide 0 if you do not want to send alert mail.</div><br> <a href=":link_url" target="_blank">Add Email Notification</a>', [':info_icon_url' => $info_icon_url, ':link_url' => $base_url . '/admin/config/people/security_login_secure/WebsiteSecurityNotifications']);

        $form['bruteforce_ip']['set_of_limit_login_features']['website_security_attack_allowed_attempts_name'] = [
            '#markup' => Markup::create($tooltip_markup),
            '#prefix' => $prefixname,
            '#suffix' => $suffixname,
        ];

        $form['bruteforce_ip']['set_of_limit_login_features']['website_security_attack_ip_allowed_attempts_value'] = [
            '#type' => 'number',
            '#min' => 0,
            '#prefix' => $prefixvalue,
            '#suffix' => $suffixvalue,
            '#default_value' => $db_var->get('website_security_ip_attack_detection')
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function submitBruteForceIpForm(array &$form, FormStateInterface $form_state)
    {
        $enable_ip_bruteforce = $form['bruteforce_ip']['website_security_ip_enable_bruteforce']['#value'];
        $db_var = $this->configFactory()->getEditable('security_login_secure.settings');

        if ($enable_ip_bruteforce == 1) {
            $db_var->set('website_security_ip_track_time', $form['bruteforce_ip']['set_of_limit_login_features']['website_security_ip_track_time_value']['#value'])
                ->set('website_security_ip_allowed_attempts', $form['bruteforce_ip']['set_of_limit_login_features']['website_security_ip_allowed_attempts_value']['#value'])
                ->set('website_security_ip_attack_detection', $form['bruteforce_ip']['set_of_limit_login_features']['website_security_attack_ip_allowed_attempts_value']['#value'])
                ->set('website_security_ip_block_time_period', $form['bruteforce_ip']['set_of_limit_login_features']['website_security_ip_blocked_time']['#value'])
                ->save();
        }

        $db_var->set('website_security_ip_enable_bruteforce', $enable_ip_bruteforce)->save();
    }
}
