<?php

namespace Drupal\security_login_secure\Form\Section;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\security_login_secure\Utilities;
use Drupal\security_login_secure\Form\WebsiteSecurityConfiguration;

/**
 * Form handler for user-based brute force protection settings.
 */
class BruteForceUserForm extends WebsiteSecurityConfiguration
{

    /**
     * {@inheritdoc}
     */
    public function buildBruteForceUserForm(array &$form, FormStateInterface $form_state)
    {
        $info_icon_url = Utilities::get_info_icon();
        $db_var = $this->config('security_login_secure.settings');
        global $base_url;
        
        $form['bruteforce_user'] = [
            '#type' => 'details',
            '#title' => $this->t('User Based Brute Force Protection'),
            '#group' => 'information',
        ];

        $form['bruteforce_user']['description'] = [
            '#type' => 'item',
            '#markup' => 'This feature protects your site from attacks by blocking user which tries to login with random usernames and passwords.<hr>',
        ];

        $form['bruteforce_user']['website_security_user_enable_bruteforce'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Enable User Based Brute Force Protection</b>'),
            '#default_value' => $db_var->get('website_security_user_enable_bruteforce')
        ];

        $prefixname = '<div class="ns_row"><div class="ns_name">';
        $suffixname = '</div>';
        $prefixvalue = '<div class="ns_value">';
        $suffixvalue = '</div></div>';

        $form['bruteforce_user']['set_of_user_limit_login_features'] = [
            '#type' => 'fieldset',
            '#attributes' => ['style' => ['padding:14px 12px 14px;color:#34495e;border-radius:4px;']],
            '#states' => [
                'visible' => [
                    ':input[name="website_security_user_enable_bruteforce"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $tooltip_markup = $this->t('Track time to check for security violations (hours) <div class="ns_tooltip"><img src=":info_icon_url" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">The time in hours for which the failed login attempts are monitored. After that time, the attempts are deleted and will never be considered again.<br> <b>Note: </b>Provide 0 if you do not want to enable the feature.</div>', [':info_icon_url' => $info_icon_url]);

        $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_user_track_time'] = [
            '#markup' => Markup::create($tooltip_markup),
            '#prefix' => $prefixname,
            '#suffix' => $suffixname,
        ];

        $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_user_track_time_value'] = [
            '#type' => 'number',
            '#min' => 0,
            '#prefix' => $prefixvalue,
            '#suffix' => $suffixvalue,
            '#default_value' => $db_var->get('website_security_user_track_time')
        ];

        $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_user_allowed_attempts_name'] = [
            '#markup' => 'Number of login failures before blocking an User <div class="ns_tooltip"><img src="' . $info_icon_url . '" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">The number of failed login attempts by an User before it gets blocked. After that count, the user will be blocked from the site until it is unblocked by admin or after the time provided below.</div>',
            '#prefix' => $prefixname,
            '#suffix' => $suffixname,
        ];

        $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_user_allowed_attempts_value'] = [
            '#type' => 'number',
            '#min' => 1,
            '#prefix' => $prefixvalue,
            '#suffix' => $suffixvalue,
            '#default_value' => $db_var->get('website_security_user_allowed_attempts')
        ];

        $tooltip_markup = $this->t('Time period for which User should be blocked (hours) <div class="ns_tooltip"><img src=":info_icon_url" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">The time in hours for which the user will remain in blocked state. After that time, the user will be unblocked.<br> <b>Note: </b>Provide 0 if you want to permanently block an user.</div>', [':info_icon_url' => $info_icon_url]);

        $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_user_blocked'] = [
            '#markup' => Markup::create($tooltip_markup),
            '#prefix' => $prefixname,
            '#suffix' => $suffixname,
        ];

        $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_user_blocked_time'] = [
            '#type' => 'number',
            '#min' => 0,
            '#prefix' => $prefixvalue,
            '#suffix' => $suffixvalue,
            '#default_value' => $db_var->get('website_security_user_block_time_period')
        ];

        $tooltip_markup = $this->t('Number of login failures before detecting an attack <div class="ns_tooltip"><img src=":info_icon_url" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">The number of failed login attempts through an IP that can be considered as an attack. After that count, the admin gets a notification email about the attack.<br> <b>Note: </b>Provide a number less than the allowed attempts or else provide 0 if you do not want to send alert mail.</div><br> <a href=":link_url" target="_blank">Add Email Notification</a>', [':info_icon_url' => $info_icon_url, ':link_url' => $base_url . '/admin/config/people/security_login_secure/WebsiteSecurityNotifications']);

        $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_attack_user_allowed_attempts_name'] = [
            '#markup' => Markup::create($tooltip_markup),
            '#prefix' => $prefixname,
            '#suffix' => $suffixname,
        ];

        $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_user_attack_allowed_attempts_value'] = [
            '#type' => 'number',
            '#min' => 0,
            '#prefix' => $prefixvalue,
            '#suffix' => $suffixvalue,
            '#default_value' => $db_var->get('website_security_user_attack_detection')
        ];

        $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_show_remaining'] = [
            '#markup' => 'Show remaining login attempts to user:',
            '#prefix' => $prefixname,
            '#suffix' => $suffixname,
        ];

        $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_show_remaining_value'] = [
            '#type' => 'checkbox',
            '#prefix' => $prefixvalue,
            '#suffix' => $suffixvalue,
            '#default_value' => $db_var->get('website_security_show_remaining_attempts')
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function submitBruteForceUserForm(array &$form, FormStateInterface $form_state)
    {
        $enable_user_bruteforce = $form['bruteforce_user']['website_security_user_enable_bruteforce']['#value'];
        $db_var = $this->configFactory()->getEditable('security_login_secure.settings');

        if ($enable_user_bruteforce == 1) {
            $db_var->set('website_security_user_track_time', $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_user_track_time_value']['#value'])
                ->set('website_security_user_allowed_attempts', $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_user_allowed_attempts_value']['#value'])
                ->set('website_security_user_attack_detection', $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_user_attack_allowed_attempts_value']['#value'])
                ->set('website_security_user_block_time_period', $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_user_blocked_time']['#value'])
                ->set('website_security_show_remaining_attempts', $form['bruteforce_user']['set_of_user_limit_login_features']['website_security_show_remaining_value']['#value'])
                ->save();
        }

        $db_var->set('website_security_user_enable_bruteforce', $enable_user_bruteforce)->save();
    }
}
