<?php

namespace Drupal\security_login_secure\Form\Section;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\security_login_secure\Utilities;
use Drupal\security_login_secure\Form\WebsiteSecurityIPBlocking;
use Drupal\security_login_secure\MiniorangeWebsiteSecurityConstants;

/**
 * Form handler for Vulnerability Scanner settings.
 */
class VulnerabilityScannerForm extends WebsiteSecurityIPBlocking
{
    /**
     * {@inheritdoc}
     */
    public function build_vulnerability_scanner_form(array &$form, FormStateInterface $form_state)
    {
        $premium_url = Url::fromRoute('security_login_secure.licensing')->toString();
        $db_var = $this->config('security_login_secure.settings');
        $info_icon_url = Utilities::get_info_icon();
        $disabled = TRUE;

        $form['vulnerability_scanner_settings'] = [
            '#type' => 'details',
            '#group' => 'information',
            '#title' => $this->t('Vulnerability Scanner'),
        ];

        $form['vulnerability_scanner_settings']['vulnerability_scanner_enabled'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Enable Vulnerability Scanner <a href=":premium_url">[Premium]</a></b> <div class="ns_tooltip"><img src=":info_icon_url" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">Enable automated vulnerability scanning to detect security issues in your Drupal site. This feature will scan for common vulnerabilities, outdated modules, and security misconfigurations.</div>', [
                ':premium_url' => $premium_url,
                ':info_icon_url' => $info_icon_url
            ]),
            '#default_value' => $db_var->get('vulnerability_scanner_enabled') ?: FALSE,
            '#disabled' => $disabled,
        ];

        $form['vulnerability_scanner_settings']['scanner_configuration'] = [
            '#type' => 'fieldset',
            '#title' => $this->t('Scanner Configuration'),
            '#attributes' => ['style' => ['color:#34495e;border-radius:4px;']],

        ];

        $form['vulnerability_scanner_settings']['scanner_configuration']['scan_frequency'] = [
            '#type' => 'select',
            '#title' => $this->t('Scan Frequency <a href=":premium_url">[Premium]</a>', [':premium_url' => $premium_url]),
            '#options' => [
                'daily' => $this->t('Daily'),
                'weekly' => $this->t('Weekly'),
                'monthly' => $this->t('Monthly'),
                'manual' => $this->t('Manual Only'),
            ],
            '#default_value' => $db_var->get('vulnerability_scan_frequency') ?: 'weekly',
            '#description' => $this->t('How often the vulnerability scanner should run automatically.'),
            '#disabled' => $disabled,
        ];

        $form['vulnerability_scanner_settings']['scanner_configuration']['scan_depth'] = [
            '#type' => 'select',
            '#title' => $this->t('Scan Depth <a href=":premium_url">[Premium]</a>', [':premium_url' => $premium_url]),
            '#options' => [
                'basic' => $this->t('Basic - Quick scan for common issues'),
                'standard' => $this->t('Standard - Comprehensive security check'),
                'deep' => $this->t('Deep - Thorough analysis including file system'),
            ],
            '#default_value' => $db_var->get('vulnerability_scan_depth') ?: 'standard',
            '#description' => $this->t('Level of detail for vulnerability scanning. Deeper scans take longer but find more issues.'),
            '#disabled' => $disabled,
        ];

        $form['vulnerability_scanner_settings']['scanner_configuration']['scan_modules'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Scan Module Vulnerabilities <a href=":premium_url">[Premium]</a></b>', [':premium_url' => $premium_url]),
            '#default_value' => $db_var->get('vulnerability_scan_modules') ?: TRUE,
            '#description' => $this->t('Check for known vulnerabilities in installed modules and themes.'),
            '#disabled' => $disabled,
        ];

        $form['vulnerability_scanner_settings']['scanner_configuration']['scan_core'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Scan Drupal Core Vulnerabilities <a href=":premium_url">[Premium]</a></b>', [':premium_url' => $premium_url]),
            '#default_value' => $db_var->get('vulnerability_scan_core') ?: TRUE,
            '#description' => $this->t('Check for known security issues in Drupal core.'),
            '#disabled' => $disabled,
        ];

        $form['vulnerability_scanner_settings']['scanner_configuration']['scan_file_permissions'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Scan File Permissions <a href=":premium_url">[Premium]</a></b>', [':premium_url' => $premium_url]),
            '#default_value' => $db_var->get('vulnerability_scan_file_permissions') ?: TRUE,
            '#description' => $this->t('Check for insecure file and directory permissions.'),
            '#disabled' => $disabled,
        ];

        $form['vulnerability_scanner_settings']['scanner_configuration']['scan_database'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Scan Database Security <a href=":premium_url">[Premium]</a></b>', [':premium_url' => $premium_url]),
            '#default_value' => $db_var->get('vulnerability_scan_database') ?: TRUE,
            '#description' => $this->t('Check for database security misconfigurations and weak credentials.'),
            '#disabled' => $disabled,
        ];

        $form['vulnerability_scanner_settings']['notification_settings'] = [
            '#type' => 'fieldset',
            '#title' => $this->t('Notification Settings'),
            '#attributes' => ['style' => ['color:#34495e;border-radius:4px;']],
        ];

        $form['vulnerability_scanner_settings']['notification_settings']['email_notifications'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Enable Email Notifications <a href=":premium_url">[Premium]</a></b>', [':premium_url' => $premium_url]),
            '#default_value' => $db_var->get('vulnerability_email_notifications') ?: FALSE,
            '#description' => $this->t('Send email alerts when vulnerabilities are detected.'),
            '#disabled' => $disabled,
        ];

        $form['vulnerability_scanner_settings']['notification_settings']['notification_email'] = [
            '#type' => 'email',
            '#title' => $this->t('Notification Email <a href=":premium_url">[Premium]</a>', [':premium_url' => $premium_url]),
            '#default_value' => $db_var->get('vulnerability_notification_email') ?: '',
            '#description' => $this->t('Email address to receive vulnerability scan reports.'),
            '#disabled' => $disabled,
        ];

        $form['vulnerability_scanner_settings']['notification_settings']['severity_threshold'] = [
            '#type' => 'select',
            '#title' => $this->t('Minimum Severity for Notifications <a href=":premium_url">[Premium]</a>', [':premium_url' => $premium_url]),
            '#options' => [
                'critical' => $this->t('Critical Only'),
                'high' => $this->t('High and Critical'),
                'medium' => $this->t('Medium, High and Critical'),
                'low' => $this->t('All Severities'),
            ],
            '#default_value' => $db_var->get('vulnerability_severity_threshold') ?: 'high',
            '#description' => $this->t('Minimum vulnerability severity level to trigger notifications.'),
            '#disabled' => $disabled,
        ];

        $form['vulnerability_scanner_settings']['action_settings'] = [
            '#type' => 'fieldset',
            '#title' => $this->t('Automatic Actions'),
            '#attributes' => ['style' => ['color:#34495e;border-radius:4px;']],
        ];

        $form['vulnerability_scanner_settings']['action_settings']['auto_block_ips'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Auto-block IPs with Critical Vulnerabilities <a href=":premium_url">[Premium]</a></b>', [':premium_url' => $premium_url]),
            '#default_value' => $db_var->get('vulnerability_auto_block_ips') ?: FALSE,
            '#description' => $this->t('Automatically block IP addresses that trigger critical vulnerability scans.'),
            '#disabled' => $disabled,
        ];

        $form['vulnerability_scanner_settings']['action_settings']['auto_update_modules'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Auto-update Vulnerable Modules <a href=":premium_url">[Premium]</a></b>', [':premium_url' => $premium_url]),
            '#default_value' => $db_var->get('vulnerability_auto_update_modules') ?: FALSE,
            '#description' => $this->t('Automatically update modules when security updates are available.'),
            '#disabled' => $disabled,
        ];

        $form['vulnerability_scanner_settings']['action_settings']['quarantine_files'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Quarantine Suspicious Files <a href=":premium_url">[Premium]</a></b>', [':premium_url' => $premium_url]),
            '#default_value' => $db_var->get('vulnerability_quarantine_files') ?: FALSE,
            '#description' => $this->t('Move suspicious files to a quarantine directory for manual review.'),
            '#disabled' => $disabled,
        ];

        // Manual scan button
        $form['vulnerability_scanner_settings']['manual_scan'] = [
            '#type' => 'fieldset',
            '#title' => $this->t('Manual Scan'),
            '#attributes' => ['style' => ['color:#34495e;border-radius:4px;']],
            '#states' => [
                'visible' => [
                    ':input[name="vulnerability_scanner_enabled"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $form['vulnerability_scanner_settings']['manual_scan']['run_scan_button'] = [
            '#type' => 'submit',
            '#value' => $this->t('Run Vulnerability Scan Now <a href=":premium_url">[Premium]</a>', [':premium_url' => $premium_url]),
            '#submit' => ['::runVulnerabilityScan'],
            '#disabled' => $disabled,
            '#attributes' => [
                'class' => ['button--secondary'],
                'title' => $this->t('This feature is disabled in the current version. Upgrade to Premium to enable vulnerability scanning.'),
            ],
        ];

        $form['vulnerability_scanner_settings']['manual_scan']['scan_progress'] = [
            '#type' => 'markup',
            '#markup' => '<div id="vulnerability-scan-progress" style="display: none;">
                <div class="progress-bar">
                    <div class="progress-fill" style="width: 0%; background-color: #007cba; height: 20px; border-radius: 3px;"></div>
                </div>
                <div class="scan-status">Scanning in progress...</div>
            </div>',
        ];

        // Scan results display
        $form['vulnerability_scanner_settings']['scan_results'] = [
            '#type' => 'fieldset',
            '#title' => $this->t('Latest Scan Results'),
            '#attributes' => ['style' => ['color:#34495e;border-radius:4px;']],
            '#states' => [
                'visible' => [
                    ':input[name="vulnerability_scanner_enabled"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $last_scan = $db_var->get('vulnerability_last_scan');
        if ($last_scan) {
            $form['vulnerability_scanner_settings']['scan_results']['last_scan_info'] = [
                '#type' => 'markup',
                '#markup' => '<div class="scan-info">
                    <p><strong>Last Scan:</strong> ' . date('Y-m-d H:i:s', $last_scan) . '</p>
                    <p><strong>Status:</strong> <span class="scan-status-completed">Completed</span></p>
                    <p><strong>Vulnerabilities Found:</strong> <span class="vulnerability-count">' . ($db_var->get('vulnerability_count') ?: 0) . '</span></p>
                </div>',
            ];
        } else {
            $form['vulnerability_scanner_settings']['scan_results']['no_scan_info'] = [
                '#type' => 'markup',
                '#markup' => '<div class="no-scan-info">
                    <p>No vulnerability scans have been performed yet.</p>
                    <p>Click "Run Vulnerability Scan Now" to perform your first scan.</p>
                </div>',
            ];
        }
    }

    /**
     * {@inheritdoc}
     */
    public function submit_vulnerability_scanner_form(array &$form, FormStateInterface $form_state)
    {
        $config = $this->configFactory()->getEditable('security_login_secure.settings');
        $account = \Drupal::currentUser();

        // Save vulnerability scanner settings
        $config->set('vulnerability_scanner_enabled', $form_state->getValue('vulnerability_scanner_enabled'))
            ->set('vulnerability_scan_frequency', $form_state->getValue('scan_frequency'))
            ->set('vulnerability_scan_depth', $form_state->getValue('scan_depth'))
            ->set('vulnerability_scan_modules', $form_state->getValue('scan_modules'))
            ->set('vulnerability_scan_core', $form_state->getValue('scan_core'))
            ->set('vulnerability_scan_file_permissions', $form_state->getValue('scan_file_permissions'))
            ->set('vulnerability_scan_database', $form_state->getValue('scan_database'))
            ->set('vulnerability_email_notifications', $form_state->getValue('email_notifications'))
            ->set('vulnerability_notification_email', $form_state->getValue('notification_email'))
            ->set('vulnerability_severity_threshold', $form_state->getValue('severity_threshold'))
            ->set('vulnerability_auto_block_ips', $form_state->getValue('auto_block_ips'))
            ->set('vulnerability_auto_update_modules', $form_state->getValue('auto_update_modules'))
            ->set('vulnerability_quarantine_files', $form_state->getValue('quarantine_files'))
            ->save();

        // Log the configuration change
        $this->miniorangeReportsRepository->mo_website_security_add_reports_entry(
            'Vulnerability Scanner Configuration Updated',
            $account->getAccountName(),
            'Configuration Change'
        );

        $this->messenger()->addMessage($this->t('Vulnerability scanner configuration has been saved successfully.'));
    }

    /**
     * Submit handler for manual vulnerability scan.
     */
    public function runVulnerabilityScan(array &$form, FormStateInterface $form_state)
    {
        // This is a placeholder for the actual scan functionality
        // In a real implementation, this would trigger the vulnerability scanning process

        $this->messenger()->addWarning($this->t('Vulnerability scanning is a premium feature. Please upgrade to enable this functionality.'));

        // Log the scan attempt
        $account = \Drupal::currentUser();
        $this->miniorangeReportsRepository->mo_website_security_add_reports_entry(
            'Manual Vulnerability Scan Attempted',
            $account->getAccountName(),
            'Scan Attempt'
        );
    }
}
