# Security Login Secure Module

## Overview
The Security Login Secure module is a comprehensive security solution for Drupal websites that provides advanced protection against brute force attacks and unauthorized access attempts. The module implements a dual-layer security system that protects both IP addresses and user accounts from malicious login attempts.

## Core Features

### 1. IP-based Security
- **IP Blocking**: Automatically blocks IP addresses after multiple failed login attempts
- **IP Whitelisting**: Allows specific IP addresses to bypass security restrictions
- **Attack Detection**: Monitors and detects potential brute force attacks
- **Customizable Attempt Limits**: Configurable number of allowed failed attempts
- **Time-based Tracking**: Configurable time window for tracking failed attempts

### 2. User-based Security
- **User Account Protection**: Blocks user accounts after multiple failed login attempts
- **Remaining Attempts Display**: Shows users their remaining login attempts
- **Attack Detection**: Monitors suspicious user login patterns
- **Automatic Unblocking**: Time-based automatic unblocking of user accounts
- **Email Notifications**: Alerts administrators about security events

### 3. Additional Security Features
- **Email Notifications**: Configurable email alerts for security events
- **Custom Error Messages**: Configurable error messages for blocked users/IPs
- **Security Reports**: Detailed logging of security events
- **Integration with Drupal Core**: Works alongside Drupal's built-in security features

## Technical Workflow

### 1. IP Security Workflow
```mermaid
graph TD
    A[Login Attempt] --> B{IP Check}
    B -->|Whitelisted| C[Allow Access]
    B -->|Not Whitelisted| D{Check Block Status}
    D -->|Blocked| E[Block Access]
    D -->|Not Blocked| F{Check Attempts}
    F -->|Under Limit| G[Allow Attempt]
    F -->|Over Limit| H[Block IP]
    H --> I[Send Notification]
```

### 2. User Security Workflow
```mermaid
graph TD
    A[Login Attempt] --> B{User Status Check}
    B -->|Blocked| C[Block Access]
    B -->|Not Blocked| D{Authentication}
    D -->|Success| E[Allow Access]
    D -->|Failure| F{Check Attempts}
    F -->|Under Limit| G[Show Remaining Attempts]
    F -->|Over Limit| H[Block User]
    H --> I[Send Notification]
```

## Technical Implementation

### Service Architecture
1. **MiniorangeIPSecurityManager**
   - Handles IP-based security checks
   - Manages IP blocking and whitelisting
   - Tracks failed login attempts by IP
   - Sends IP-related security notifications

2. **MiniorangeUserSecurityManager**
   - Manages user account security
   - Handles user blocking and unblocking
   - Tracks failed login attempts by user
   - Manages user-related security notifications

### Repository Layer
1. **MiniorangeIPSecurityRepository**
   - Database operations for IP security
   - IP blocking/unblocking operations
   - IP attempt tracking

2. **MiniorangeUserSecurityRepository**
   - Database operations for user security
   - User blocking/unblocking operations
   - User attempt tracking

## Frequently Asked Questions (FAQs)

### 1. How to Check if a User is Blocked?
```php
// Using the service
$userSecurityManager = \Drupal::service('security_login_secure.user_security_manager');
$isBlocked = $userSecurityManager->isUserBlocked($username);
```

### 2. How to Unblock a User Manually?
```php
// Using the repository
$userSecurityRepository = \Drupal::service('security_login_secure.user_security_repository');
$userSecurityRepository->mo_website_security_clear_events_users($ip_address, $username, $account);
```

### 3. How to Check if an IP is Blocked?
```php
// Using the service
$ipSecurityManager = \Drupal::service('security_login_secure.ip_security_manager');
$isBlocked = $ipSecurityManager->isIpBlocked($ip_address);
```

### 4. How to Trace Back Block User Issues?
1. Check the security reports in the admin interface
2. Review the database tables:
   - `mo_website_security_user_attempts`
   - `mo_website_security_ip_attempts`
3. Check the Drupal logs for security-related entries
4. Review email notifications sent by the system

### 5. How to Configure Security Settings?
1. Navigate to `/admin/config/security/security-login-secure`
2. Configure:
   - Allowed attempts for IPs and users
   - Tracking time windows
   - Email notification settings
   - Whitelist/blacklist IPs

### 6. How to Handle False Positives?
1. Use IP whitelisting for trusted IPs
2. Adjust the allowed attempts threshold
3. Modify the tracking time window
4. Manually unblock users/IPs if needed

### 7. How to Monitor Security Events?
1. Check the security reports page
2. Monitor email notifications
3. Review the Drupal logs
4. Use the attack detection feature

## Best Practices

1. **Configuration**
   - Set reasonable attempt limits
   - Configure appropriate tracking times
   - Enable email notifications
   - Maintain an updated IP whitelist

2. **Monitoring**
   - Regularly check security reports
   - Monitor failed login attempts
   - Review blocked users/IPs
   - Keep track of attack notifications

3. **Maintenance**
   - Regularly review and update whitelisted IPs
   - Clean up old security logs
   - Update the module regularly
   - Monitor system performance

## Troubleshooting

### Common Issues and Solutions

1. **User Cannot Login**
   - Check if the user is blocked
   - Verify IP is not blocked
   - Check remaining attempts
   - Review security logs

2. **False Blocking**
   - Adjust attempt limits
   - Modify tracking time
   - Add IP to whitelist
   - Manually unblock user/IP

3. **Email Notifications Not Working**
   - Verify email configuration
   - Check spam folders
   - Verify notification settings
   - Check server mail settings 