<?php

namespace Drupal\security_login_secure\Form\Section;

use Drupal\Core\Form\FormStateInterface;
use Drupal\security_login_secure\Utilities;
use Drupal\security_login_secure\Form\WebsiteSecurityIPBlocking;
use Drupal\user\Entity\User;
use Drupal\security_login_secure\MiniorangeWebsiteSecurityConstants;

/**
 * Form handler for IP blocking settings.
 */
class IPBlockingForm extends WebsiteSecurityIPBlocking
{
    /**
     * {@inheritdoc}
     */
    public function build_ip_blocking_form(array &$form, FormStateInterface $form_state)
    {
        global $base_url;
        $db_var = $this->config('security_login_secure.settings');
        $disabled = TRUE;
        $info_icon_url = Utilities::get_info_icon();

        $form['blocking_whitelisting'] = [
            '#type' => 'details',
            '#title' => 'Blocking / Whitelisting',
            '#group' => 'information',
        ];

        $form['blocking_whitelisting']['website_security_enable_ip_blocking'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Enable IP Blocking</b>'),
            '#default_value' => $db_var->get('website_security_enable_ip_blocking')
        ];

        $form['blocking_whitelisting']['block_ip_list'] = array(
            '#type' => 'fieldset',
            '#attributes' => array('style' => array('color:#34495e;border-radius:4px;')),
            '#states' => array(
                'visible' => array(
                    ':input[name="website_security_enable_ip_blocking"]' => array('checked' => True),
                ),
            ),
        );

        $form['blocking_whitelisting']['block_ip_list']['markup'] = [
            '#markup' => '<h5>Manual Block IPs</h5>'
        ];


        $form['blocking_whitelisting']['block_ip_list']['manual_block_ip_address'] = [
            '#type' => 'textarea',
            '#attributes' => ['placeholder' => $this->t('semicolon(;) separated IP address')],
            '#default_value' => $db_var->get('website_security_block_ips'),
            '#description' => $this->t('<b>Note: </b>Provide semicolon(;) separated IP address that will be blocked from the site.')
        ];

        $form['blocking_whitelisting']['website_security_enable_ip_whitelisting'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Enable IP Whitelisting</b>'),
            '#default_value' => $db_var->get('website_security_enable_ip_whitelisting')
        ];

        $form['blocking_whitelisting']['whitelist_ip_list'] = array(
            '#type' => 'fieldset',
            '#attributes' => array('style' => array('color:#34495e;border-radius:4px;')),
            '#states' => array(
                'visible' => array(
                    ':input[name="website_security_enable_ip_whitelisting"]' => array('checked' => True),
                ),
            ),
        );

        $form['blocking_whitelisting']['whitelist_ip_list']['markup'] = [
            '#markup' => '<h5>Whitelist IPs</h5>'
        ];

        $form['blocking_whitelisting']['whitelist_ip_list']['manual_whitelist_ip_address'] = [
            '#type' => 'textarea',
            '#attributes' => ['placeholder' => $this->t('semicolon(;) separated IP address')],
            '#default_value' => $db_var->get('website_security_white_ips'),
            '#description' => $this->t('<b>Note: </b>Provide semicolon(;) separated IP address that will be whitelisted and will never be blocked in any case.')
        ];

        $form['blocking_whitelisting']['whitelist_ip_list']['whitelist_user'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Enable User whitelisting if IP is whitelisted</b> <div class="ns_tooltip"><img src=":info_icon_url" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">Enable this feature if you do not want your site users to be blocked if the corresponding IP is whitelisted.<br> Note: If you do not enable this feature then site users will be blocked after 5 invalid attempts (Drupal default behaviour) even if IP is whitelisted.</div>', [':info_icon_url' => $info_icon_url]),
            '#default_value' => $db_var->get('website_security_disable_user_blocking')
        ];

        $form['blocking_whitelisting']['website_security_enforce_strong_password'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Enforce Strong Password <a href=":premium_url">[Premium]</a></b> <div class="ns_tooltip"><img src=":info_icon_url" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">Enable this feature if you want to enforce your users to save strong password while registration. A strong password is difficult to be guessed using manual and automatic password cracking tools.</div>', [':premium_url' => $base_url . MiniorangeWebsiteSecurityConstants::LICENSING_TAB_URL, ':info_icon_url' => $info_icon_url]),
            '#disabled' => $disabled,
        ];

        $form['blocking_whitelisting']['website_security_enable_bot_detection'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Enable Bot Blocking <a href=":premium_url">[Premium]</a></b> <div class="ns_tooltip"><img src=":info_icon_url" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">Check this feature if you want to enable bot detection. A botnet is a network of compromised computers under the control of a malicious actor. Each individual device in a botnet is referred to as a bot. A bot is formed when a computer gets infected with malware that enables third-party control.</div>', [':premium_url' => $base_url . MiniorangeWebsiteSecurityConstants::LICENSING_TAB_URL, ':info_icon_url' => $info_icon_url]),
            '#disabled' => $disabled,
        ];

        $form['blocking_whitelisting']['website_security_dos_protection'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Enable DoS Protection <a href=":premium_url">[Premium]</a></b> <div class="ns_tooltip"><img src=":info_icon_url" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">A denial-of-server attack is an explicit attempt to deny users from using a service or computer resource. Enable DoS protection feature can filter suspicious or unreasonable packets to prevent from flooding the network with large amounts of fake traffic.</div>', [':premium_url' => $base_url . MiniorangeWebsiteSecurityConstants::LICENSING_TAB_URL, ':info_icon_url' => $info_icon_url]),
            '#disabled' => $disabled,
        ];

        $form['blocking_whitelisting']['website_security_risk_based_authentication'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('<b>Enable Risk Based Authentication <a href=":premium_url">[Premium]</a></b> <div class="ns_tooltip"><img src=":info_icon_url" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">Risk-based implementation allows the application to challenge the user for additional credentials only when the risk level is appropriate. Enable this feature if you want to allow risk-based authentication.</div>', [':premium_url' => $base_url . MiniorangeWebsiteSecurityConstants::LICENSING_TAB_URL, ':info_icon_url' => $info_icon_url]),
            '#disabled' => $disabled,
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function submit_ip_blocking_form(array &$form, FormStateInterface $form_state)
    {
        $config = $this->configFactory()->getEditable('security_login_secure.settings');
        $account = User::load(\Drupal::currentUser()->id());
        $enable_ip_blocking = $form['blocking_whitelisting']['website_security_enable_ip_blocking']['#value'];
        $enable_ip_whitelisting = $form['blocking_whitelisting']['website_security_enable_ip_whitelisting']['#value'];
        $whitelist_user = $form['blocking_whitelisting']['whitelist_ip_list']['whitelist_user']['#value'];

        if ($enable_ip_blocking == 1) {
            $block_ips = filter_var(trim($form['blocking_whitelisting']['block_ip_list']['manual_block_ip_address']['#value']));
            if (empty($block_ips)) {
                $this->messenger()->addError($this->t('IP address for blocking is required.'));
                \Drupal::state()->set('warning_in_saving_config', TRUE);
                return;
            }
            $block_ips = $this->website_security_validate_ip_address($block_ips, 'blocking');
        }

        if ($enable_ip_whitelisting == 1) {
            $whitelist_ips = filter_var(trim($form['blocking_whitelisting']['whitelist_ip_list']['manual_whitelist_ip_address']['#value']));
            if (empty($whitelist_ips)) {
                $this->messenger()->addError($this->t('IP address for whitelist is required.'));
                \Drupal::state()->set('warning_in_saving_config', TRUE);
                return;
            }
            $whitelist_ips = $this->website_security_validate_ip_address($whitelist_ips, 'whitelisting');
        }

        if ($enable_ip_blocking == 1 && $enable_ip_whitelisting == 1) {
            $blocked_ips = explode(';', $block_ips);
            $whitelisted_ips = explode(';', $whitelist_ips);
            $common_ips = array_intersect($blocked_ips, $whitelisted_ips);

            if (count($common_ips) > 0) {
                $block_ips = implode(';', array_diff($blocked_ips, $whitelisted_ips));
                $whitelist_ips = implode(';', array_diff($whitelisted_ips, $blocked_ips));
                $common_ips = implode(',', $common_ips);
                $this->messenger()->addWarning($this->t('The following IPs :common_ips can either be blocked or whitelisted and hence not saved.', [':common_ips' => $common_ips]));
                \Drupal::state()->set('warning_in_saving_config', TRUE);
            }

            $config->set('website_security_block_ips', $block_ips)
                ->set('website_security_white_ips', $whitelist_ips)
                ->save();

            $this->miniorangeReportsRepository->mo_website_security_add_reports_entry($whitelist_ips, $account->getAccountName(), 'IP Whitelisted');
            $this->miniorangeReportsRepository->mo_website_security_add_reports_entry($block_ips, $account->getAccountName(), 'IP Blacklisted');
        } elseif ($enable_ip_blocking == 1) {
            $config->set('website_security_block_ips', $block_ips)->save();
            $this->miniorangeReportsRepository->mo_website_security_add_reports_entry($block_ips, $account->getAccountName(), 'IP Blacklisted');
        } elseif ($enable_ip_whitelisting == 1) {
            $config->set('website_security_white_ips', $whitelist_ips)->save();
            $this->miniorangeReportsRepository->mo_website_security_add_reports_entry($whitelist_ips, $account->getAccountName(), 'IP Whitelisted');
        }

        $config->set('website_security_enable_ip_blocking', $enable_ip_blocking)
            ->set('website_security_enable_ip_whitelisting', $enable_ip_whitelisting)
            ->set('website_security_disable_user_blocking', $whitelist_user)
            ->save();
    }

    function website_security_validate_ip_address($ip_addresses, $type)
    {
        $ip_addresses = array_filter(array_unique(explode(';', $ip_addresses)));
        $valid_ips = array();
        $invalid_ips = array();
        foreach ($ip_addresses as $ip_address) {

            if ($type == 'blocking' && $ip_address == \Drupal::request()->getClientIp()) {
                $this->messenger()->addWarning($this->t('You cannot block your own IP address i.e. :ip_address .', [':ip_address' => $ip_address]));
                \Drupal::state()->set('warning_in_saving_config', TRUE);
                continue;
            }

            if (!filter_var($ip_address, FILTER_VALIDATE_IP))
                array_push($invalid_ips, $ip_address);
            else
                array_push($valid_ips, $ip_address);
        }

        if (count($invalid_ips) > 0) {
            $invalid_ips = implode(',', $invalid_ips);
            $this->messenger()->addWarning($this->t('The following IP address <i> :invalid_ips </i> for :type is not valid and hence not saved.', [':invalid_ips' => $invalid_ips, ':type' => $type]));
            \Drupal::state()->set('warning_in_saving_config', TRUE);
        }

        return implode(';', $valid_ips);
    }
}
