<?php

/**
 * @file
 * Contains \Drupal\security_login_secure\Form\WebsiteSecurityBlockedIPAndUsers.
 */

namespace Drupal\security_login_secure\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\FormBase;
use Drupal\security_login_secure\Utilities;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\user\Entity\User;
use Drupal\security_login_secure\Repository\MiniorangeReportsRepository;
use Drupal\security_login_secure\Repository\MiniorangeIPSecurityRepository;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Pager\PagerManagerInterface;

class WebsiteSecurityBlockedIPAndUsers extends FormBase
{
    /**
     * @var MiniorangeReportsRepository
     */
    protected $miniorangeReportsRepository;

    /**
     * @var MiniorangeIPSecurityRepository
     */
    protected $miniorangeIPSecurityRepository;

    /**
     * @var PagerManagerInterface
     */
    protected $pagerManager;

    /**
     * Constructor.
     */
    public function __construct(MiniorangeReportsRepository $miniorangeReportsRepository, MiniorangeIPSecurityRepository $miniorangeIPSecurityRepository, PagerManagerInterface $pagerManager)
    {
        $this->miniorangeReportsRepository = $miniorangeReportsRepository;
        $this->miniorangeIPSecurityRepository = $miniorangeIPSecurityRepository;
        $this->pagerManager = $pagerManager;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        return new static(
            $container->get('security_login_secure.reports_repository'),
            $container->get('security_login_secure.ip_security_repository'),
            $container->get('pager.manager')
        );
    }

    public function getFormId()
    {
        return 'mo_website_security_blocked_ip_users';
    }

    public function buildForm(array $form, FormStateInterface $form_state)
    {
        global $base_url;

        $config = $this->config('security_login_secure.settings');

        // Handle unblock action
        if (isset($_GET['delete'])) {
            $this->miniorangeIPSecurityRepository->mo_website_security_unblock_ip($_GET['delete']);
        }

        $form['blocked_items_tab'] = array(
            '#attached' => array('library' => 'security_login_secure/security_login_secure.admin',),
            '#markup' => '<div class="ns_table_layout_1"><div class="ns_table_layout ns_container">',
        );

        $form['blocked_items_header'] = array(
            '#markup' => '<h2>Blocked IPs and Users</h2>',
            '#prefix' => '<div class="ns_row"><div class="ns_name">',
            '#suffix' => '</div></div><hr>',
        );

        // Drupal pagination setup
        $items_per_page = 25; // Default items per page
        $current_page = \Drupal::request()->query->get('page', 0);

        // Get all data and filter for blocked items only
        $order = 'DESC'; // Default order
        $all_data = $this->miniorangeReportsRepository->mo_website_security_return_search_values($order);

        // Filter to only show IP Blocked and User Blocked records
        $data = array_filter($all_data, function ($record) {
            return in_array($record->status, ['IP Blocked', 'User Blocked']);
        });

        $total_rows = count($data);

        // Initialize pager
        $this->pagerManager->createPager($total_rows, $items_per_page);

        // Calculate offset for current page
        $offset = $current_page * $items_per_page;

        // Get data for current page
        $paged_data = array_slice($data, $offset, $items_per_page);

        $db_rows = [];
        if (!empty($paged_data)) {
            foreach ($paged_data as $record) {
                $status_row = $this->t("<span class='failure_report'>" . $record->status . "</span>");

                $timestamp = $record->timestamp;
                $date = date('d-m-Y', $timestamp);
                $time = date('H:i:s', $timestamp);

                $url = $base_url . '/admin/config/people/security_login_secure/WebsiteSecurityBlockedIPAndUsers';
                $account = user_load_by_name($record->uname);
                if ($account) {
                    $account_id = $account->id();
                }
                // Action button logic
                $action = '--';
                if ($record->status == 'IP Blocked' && $this->miniorangeIPSecurityRepository->mo_website_security_is_ip_blocked($record->ip_address)) {
                    $action = $this->t('<a class="ns_btn2 button--primary" href=":url/?delete=:ip_address">Unblock IP</a>', [':ip_address' => $record->ip_address, ':url' => $url]);
                } elseif ($record->status == 'IP Blocked' && !$this->miniorangeIPSecurityRepository->mo_website_security_is_ip_blocked($record->ip_address)) {
                    $action = $this->t('<span class="blocked-ip">IP Unblocked</span>');
                } elseif ($record->status == 'User Blocked' && $account->status->value == 0) {
                    $action = $this->t('<a class="ns_btn2 button--primary" href=":base_url/user/:uid/edit" target="_blank">Unblock User</a>', [':base_url' => $base_url, ':uid' => $account_id]);
                } elseif ($record->status == 'User Blocked') {
                    $action = $this->t('<span class="blocked-user">User Unblocked</span>');
                }

                $db_rows[] = [
                    'ip_address' => $record->ip_address,
                    'username' => $record->uname,
                    'status' => $status_row,
                    'datetime' => $date . ' | ' . $time,
                    'operations' => $action,
                ];
            }
        }

        $form['blocked_items_table'] = array(
            '#type' => 'table',
            '#header' => [
                'IP Address',
                'Username',
                'Status',
                'Date & Time',
                'Action',
            ],
            '#rows' => $db_rows,
            '#empty' => $this->t('No blocked items found.'),
            '#attributes' => [
                'class' => ['responsive-enabled'],
            ],
        );

        // Show current page information
        $current_start = ($current_page * $items_per_page) + 1;
        $current_end = min(($current_page + 1) * $items_per_page, $total_rows);

        if ($total_rows > 0) {
            $form['showing_entries'] = [
                '#markup' => $this->t('Showing @start - @end of @total blocked items', [
                    '@start' => $current_start,
                    '@end' => $current_end,
                    '@total' => $total_rows,
                ]),
                '#prefix' => '<div class="table-info" style="margin: 10px 0; clear: both;">',
                '#suffix' => '</div>',
            ];
        }

        // Add Drupal's native pager right after the table
        $form['pager'] = [
            '#type' => 'pager',
            '#prefix' => '<div class="pager-wrapper" style="clear: both; margin: 20px 0; text-align: center;">',
            '#suffix' => '</div>',
        ];

        $form['container_end'] = array(
            '#markup' => '</div></div>',
        );

        return $form;
    }

    public function submitForm(array &$form, FormStateInterface $form_state) {}
}
