<?php

/**
 * @file
 * Contains \Drupal\security_login_secure\Form\WebsiteSecurityReports.
 */

namespace Drupal\security_login_secure\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\FormBase;
use Drupal\security_login_secure\Utilities;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\user\Entity\User;
use Drupal\security_login_secure\Repository\MiniorangeReportsRepository;
use Drupal\security_login_secure\Repository\MiniorangeIPSecurityRepository;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Pager\PagerManagerInterface;

class WebsiteSecurityReports extends FormBase
{
    /**
     * @var MiniorangeReportsRepository
     */
    protected $miniorangeReportsRepository;

    /**
     * @var MiniorangeIPSecurityRepository
     */
    protected $miniorangeIPSecurityRepository;

    /**
     * @var PagerManagerInterface
     */
    protected $pagerManager;

    /**
     * Constructor.
     */
    public function __construct(MiniorangeReportsRepository $miniorangeReportsRepository, MiniorangeIPSecurityRepository $miniorangeIPSecurityRepository, PagerManagerInterface $pagerManager)
    {
        $this->miniorangeReportsRepository = $miniorangeReportsRepository;
        $this->miniorangeIPSecurityRepository = $miniorangeIPSecurityRepository;
        $this->pagerManager = $pagerManager;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        return new static(
            $container->get('security_login_secure.reports_repository'),
            $container->get('security_login_secure.ip_security_repository'),
            $container->get('pager.manager')
        );
    }

    public function getFormId()
    {
        return 'mo_website_security_reports';
    }

    public function buildForm(array $form, FormStateInterface $form_state)
    {

        global $base_url;
        $info_icon_url = Utilities::get_info_icon();

        $config = $this->config('security_login_secure.settings');
        $db_var = \Drupal::configFactory()->getEditable('security_login_secure.settings');

        $dat = $this->miniorangeReportsRepository->mo_website_security_return_search_values();
        $disable_download_and_clear = count($dat) == 0 ? TRUE : FALSE;

        if (isset($_GET['delete'])) {
            $this->miniorangeIPSecurityRepository->mo_website_security_unblock_ip($_GET['delete']);
        }

        $form['website_security_reports_tab'] = array(
            '#attached' => array('library' => 'security_login_secure/security_login_secure.admin',),
            '#markup' => '<div class="ns_table_layout_1"><div class="ns_table_layout ns_container">',
        );

        $form['website_security_clear_report_header'] = array(
            '#markup' => '<h2> User Transactions Report </h2>',
            '#prefix' => '<div class="ns_row"><div class="ns_name">',
            '#suffix' => '</div>',
        );

        $form['website_security_download_reports'] = array(
            '#type' => 'submit',
            '#value' => $this->t('Download Reports'),
            '#submit' => array('::website_security_download_reports'),
            '#disabled' => $disable_download_and_clear,
            '#prefix' => '<div class="ns_download">',
            '#suffix' => '</div>',
            '#attributes' => array(
                'class' => ['button--primary'],
            ),
        );

        $form['website_security_clear_reports'] = array(
            '#type' => 'submit',
            '#value' => $this->t('Clear Reports'),
            '#submit' => array('::website_security_clear_reports'),
            '#disabled' => $disable_download_and_clear,
            '#attributes' => array(
                'onclick' => 'return confirm("Are you sure you want to clear all reports? This action cannot be undone.");',
                'class' => ['button--danger'],
            ),
            '#prefix' => '<div class="ns_clear">',
            '#suffix' => '</div></div><hr>',
        );

        $form['username_value'] = [
            '#type' => 'textfield',
            '#title' => 'Username',
            '#placeholder' => 'Enter username (Optional)',
            '#default_value' => $config->get('website_security_username_value'),
            '#prefix' => '<div class="ns_search_row"><div class="ns_search_name">',
            '#suffix' => '</div>',
        ];

        $form['ip_value'] = [
            '#type' => 'textfield',
            '#title' => 'IP Address',
            '#placeholder' => 'Enter IP address (Optional)',
            '#default_value' => $config->get('website_security_ip_value'),
            '#prefix' => '<div class="ns_search_ip">',
            '#suffix' => '</div>',
        ];

        $options = [
            'All' => 'All',
            'Success' => 'Success',
            'IP Login Failed' => 'IP Login Failed',
            'User Login Failed' => 'User Login Failed',
            'IP Whitelisted' => 'IP Whitelisted',
            'IP Blacklisted' => 'IP Blacklisted',
            'IP Blocked' => 'IP Blocked',
            'User Blocked' => 'User Blocked',
        ];

        $form['status_value'] = [
            '#type' => 'select',
            '#title' => $this->t('Status <div class="ns_tooltip"><img src=":info_icon_url" alt="info icon" height="20px" width="15px"></div><div class="ns_tooltiptext">Success => All the successful login entries <br> IP Login Failed => Failed login entries of IPs <br> User Login Failed => Failed login entries of site users <br> IP Whitelisted => Manually added IPs for whitelisting <br> IP Blacklisted => Manually added IP for blocking <br> IP Blocked => IP blocked due to invalid login attempts <br> User Blocked => Site user blocked due to invalid login attempts</div>', [
                ':info_icon_url' => $info_icon_url,
            ]),
            '#options' => $options,
            '#default_value' => $config->get('website_security_status_value'),
            '#attributes' => array('style' => 'width:90% !important',),
            '#prefix' => '<div class="ns_search_status">',
            '#suffix' => '</div>',
        ];

        $form['search'] = [
            '#type' => 'submit',
            '#submit' => array('::website_security_advanced_search'),
            '#value' => $this->t('Search'),
            '#suffix' => '</div>',
        ];

        // Drupal pagination setup
        $items_per_page = 25; // Default items per page
        $current_page = \Drupal::request()->query->get('page', 0);

        // Get all data
        $order = 'DESC'; // Default order
        $data = $this->miniorangeReportsRepository->mo_website_security_return_search_values($order);
        $total_rows = count($data);

        // Initialize pager
        $this->pagerManager->createPager($total_rows, $items_per_page);

        // Calculate offset for current page
        $offset = $current_page * $items_per_page;

        // Get data for current page
        $paged_data = array_slice($data, $offset, $items_per_page);

        $db_rows = [];
        if (!empty($paged_data)) {
            foreach ($paged_data as $record) {
                $status_row = $this->t("<span class='failure_report'>" . $record->status . "</span>");
                if ($record->status == 'IP Whitelisted' || $record->status == 'Success' || $record->status == 'Unblocked User' || $record->status == 'Unblocked IP') {
                    $status_row = $this->t("<span class='success_report'>" . $record->status . "</span>");
                }
                $timestamp = $record->timestamp;
                $date = date('d-m-Y', $timestamp);
                $time = date('H:i:s', $timestamp);

                $db_rows[] = [
                    'ip_address' => $record->ip_address,
                    'username' => $record->uname,
                    'status' => $status_row,
                    'datetime' => $date . ' | ' . $time,
                ];
            }
        }

        $form['miniorange_ns_report_table_start'] = array(
            '#type' => 'table',
            '#header' => [
                'IP Address',
                'Username',
                'Status',
                'Date & Time'
            ],
            '#rows' => $db_rows,
            '#empty' => $this->t('No records found.'),
            '#attributes' => [
                'class' => ['responsive-enabled'],
            ],
        );

        // Show current page information
        $current_start = ($current_page * $items_per_page) + 1;
        $current_end = min(($current_page + 1) * $items_per_page, $total_rows);

        if ($total_rows > 0) {
            $form['showing_entries'] = [
                '#markup' => $this->t('Showing @start - @end of @total entries', [
                    '@start' => $current_start,
                    '@end' => $current_end,
                    '@total' => $total_rows,
                ]),
                '#prefix' => '<div class="table-info" style="margin: 10px 0; clear: both;">',
                '#suffix' => '</div>',
            ];
        }

        // Add Drupal's native pager right after the table
        $form['pager'] = [
            '#type' => 'pager',
            '#prefix' => '<div class="pager-wrapper" style="clear: both; margin: 20px 0; text-align: center;">',
            '#suffix' => '</div>',
        ];

        $form['aa'] = array(
            '#markup' => '</div>',
        );

        $form['main_layout_div_end_1'] = array(
            '#markup' => '<div>',
        );

        return $form;
    }

    public function submitForm(array &$form, FormStateInterface $form_state) {}


    /**
     * Download CSV file of the report
     */
    function website_security_download_reports(&$form, $form_state)
    {

        $data = $this->miniorangeReportsRepository->mo_website_security_return_search_values();

        $reports = "S.NO,IP ADDRESS,USERNAME,REASON,DATE & TIME\n";
        $i = 1;

        foreach ($data as $value) {
            $timestamp = $value->timestamp;
            $date = date('d-m-Y H:i:s', $timestamp);
            $reports .= $i . ',' . $value->ip_address . ',' . $value->uname . ',' . $value->status . ',' . $date . "\n";
            $i++;
        }

        header('Content-Type: application/csv');
        header('Content-Disposition: attachment; filename="reports.csv"');

        print_r($reports);
        exit();
    }

    /**
     * Delete the records from the DB table
     */
    function website_security_clear_reports(&$form, $form_state)
    {
        global $base_url;
        $db = \Drupal::database();

        // Get all records with 'IP Blocked' status
        $blocked_records = $db->select('miniorange_website_security_reports', 'r')
            ->fields('r', ['ip_address'])
            ->condition('status', 'IP Blocked')
            ->execute()
            ->fetchAll();

        // Check which IPs are still actually blocked
        $still_blocked_ips = [];
        foreach ($blocked_records as $record) {
            if ($this->miniorangeIPSecurityRepository->mo_website_security_is_ip_blocked($record->ip_address)) {
                $still_blocked_ips[] = $record->ip_address;
            }
        }

        // Delete all records except those where IP is still blocked
        if (!empty($still_blocked_ips)) {
            // Delete records that are NOT 'IP Blocked' status
            $db->delete('miniorange_website_security_reports')
                ->condition('status', 'IP Blocked', '!=')
                ->execute();

            // Delete 'IP Blocked' records where IP is no longer actually blocked
            $db->delete('miniorange_website_security_reports')
                ->condition('status', 'IP Blocked')
                ->condition('ip_address', $still_blocked_ips, 'NOT IN')
                ->execute();
        } else {
            // If no IPs are still blocked, delete all records
            $db->delete('miniorange_website_security_reports')
                ->execute();
        }

        $url = $base_url . '/admin/config/people/security_login_secure/WebsiteSecurityReports';
        $response = new RedirectResponse($url);
        $response->send();
    }

    function website_security_advanced_search(&$form, $form_state)
    {
        global $base_url;
        $config = $this->configFactory()->getEditable('security_login_secure.settings');
        $username_value = $form_state->getValue('username_value');
        $ip_value = $form_state->getValue('ip_value');
        $status_value = $form_state->getValue('status_value');
        $url = $base_url . '/admin/config/people/security_login_secure/WebsiteSecurityReports';

        if (!filter_var($ip_value, FILTER_VALIDATE_IP))
            $ip_value = '';

        $config->set('website_security_ip_value', $ip_value)
            ->set('website_security_username_value', $username_value)
            ->set('website_security_status_value', $status_value)
            ->save();

        $response = new RedirectResponse($url);
        $response->send();
    }
}
