<?php

namespace Drupal\security_login_secure\Repository;

use Drupal\Core\Database\Connection;
use Drupal\user\Entity\User;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\security_login_secure\Repository\MiniorangeReportsRepository;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

class MiniorangeIPSecurityRepository
{
    /**
     * The database connection.
     *
     * @var Connection
     */
    protected $connection;

    /**
     * The Miniorange Reports Repository
     * 
     * @var MiniorangeReportsRepository
     */
    protected $miniorangeReportsRepository;

    /**
     * The messenger service.
     *
     * @var \Drupal\Core\Messenger\MessengerInterface
     */
    protected $messenger;
    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * Construct a repository object.
     *
     * @param Connection $connection
     *   The database connection.
     * @param MiniorangeReportsRepository $miniorangeReportsRepository
     *   The Miniorange Reports Repository
     * @param \Drupal\Core\Messenger\MessengerInterface $messenger
     *   The messenger service.
     * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
     *   The entity type manager.
     */
    public function __construct(
        Connection $connection,
        MiniorangeReportsRepository $miniorangeReportsRepository,
        MessengerInterface $messenger,
        EntityTypeManagerInterface $entity_type_manager
    ) {
        $this->connection = $connection;
        $this->miniorangeReportsRepository = $miniorangeReportsRepository;
        $this->messenger = $messenger;
        $this->entityTypeManager = $entity_type_manager;
    }

    /**
     * To check if IP address is blocked
     * 
     * @param string ip_address
     */
    public function mo_website_security_is_ip_blocked($ip_address)
    {
        $ip_exists = $this->connection->select('miniorange_website_security_ip_track', 'ip_address')
            ->fields('ip_address')
            ->condition('ip_address', $ip_address)
            ->countQuery()
            ->execute()
            ->fetchField();

        if ($ip_exists == 0)
            return FALSE;

        return TRUE;
    }


    /**
     * Unblock the IP address once the blocking time completes
     * 
     * @param string ip_address
     * @param string username
     * @param boolean is_blocked
     */
    public function mo_website_security_clear_events_ips($ip_address, $username, $is_blocked = true)
    {
        $ip_type = 'blocked';
        $by_admin = 0;
        $deleted = 0;
        $db_var = \Drupal::config('security_login_secure.settings');
        $blocked_time_period = $db_var->get('website_security_ip_block_time_period');
        if ($is_blocked == true) {

            if ($blocked_time_period > 0) {
                $blocked_time_period *= 3600;
                $time = \Drupal::time()->getRequestTime() - $blocked_time_period;

                $deleted = $this->connection->delete('miniorange_website_security_ip_track')
                    ->condition('blocked_timestamp', $time, '<')
                    ->condition('ip_address', $ip_address, '=')
                    ->condition('ip_type', $ip_type, '=')
                    ->condition('by_admin', $by_admin, '=')
                    ->execute();

                if ($deleted) {
                    $this->miniorangeReportsRepository->mo_website_security_add_reports_entry($ip_address, $username, 'Unblocked IP');
                    website_security_delete_session_values();
                }
            }
        } else {
            $deleted = $this->connection->delete('miniorange_website_security_ip_track')
                ->condition('ip_address', $ip_address, '=')
                ->condition('by_admin', $by_admin, '=')
                ->execute();
        }
    }

    /**
     * Check if the current client's IP address is blocked
     * 
     * @param string ip_address
     */
    public function mo_website_security_is_current_ip_blocked($ip_address)
    {
        $ip_exists = $this->connection->select('miniorange_website_security_ip_track', 'ns_ip')
            ->fields('ns_ip', ['by_admin', 'ip_address', 'ip_type'])
            ->condition('ip_address', $ip_address)
            ->execute()
            ->fetchAll();

        if ((count($ip_exists) > 0 && $ip_exists[0]->ip_type == 'blocked'))
            return 1;

        return 0;
    }

    /**
     * Check if IP address is whitelisted 
     * 
     * @param string ip_address
     */
    public function mo_website_security_is_ip_whitelisted($ip_address)
    {
        $db_var = \Drupal::config('security_login_secure.settings');
        $enable_ip_whitelisting = $db_var->get('website_security_enable_ip_whitelisting');
        if ($enable_ip_whitelisting == 1) {
            $whitelist_ips = $db_var->get('website_security_white_ips');
            $whitelist_ips = explode(';', $whitelist_ips);

            if (array_search($ip_address, $whitelist_ips) === FALSE) {
                return FALSE;
            }

            return TRUE;
        }

        return FALSE;
    }

    /**
     * Flushes the Drupal default table of logging failed attempts to override the blocking logic
     */
    public function mo_website_security_clear_flood_table_events_ip()
    {
        if ($this->connection->schema()->tableExists('flood')) {
            try {
                $this->connection->delete('flood')
                    ->condition('event', 'user.failed_login_ip', '=')
                    ->execute();
            } catch (\Exception $exception) {
                $this->messenger->addError(t('Something went wrong. Please see your recent log messages for details'));
                \Drupal::logger('security_login_secure.info')->info($exception);
            }
        }
    }

    /**
     * Returns no of failed attempts from given IP address
     * 
     * @param string ip_address
     */
    public function mo_website_security_get_ip_attempts($ip_address)
    {
        $ip_attempts = 0;
        $ip_exists = $this->connection->select('miniorange_website_security_ip_track', 'ip_address')
            ->fields('ip_address')
            ->condition('ip_address', $ip_address)
            ->countQuery()
            ->execute()
            ->fetchField();

        if ($ip_exists == 0) {
            $this->connection->insert('miniorange_website_security_ip_track')
                ->fields([
                    'ip_address' => $ip_address,
                    'failed_timestamp' => \Drupal::time()->getRequestTime(),
                ])
                ->execute();
        } else {
            $ip_attempts = $this->connection->select('miniorange_website_security_ip_track', 'ns_ip')
                ->fields('ns_ip', ['login_attempts'])
                ->condition('ip_address', $ip_address)
                ->execute()
                ->fetchField();
        }

        return $ip_attempts;
    }

    /**
     * Log the given IP address in DB table to track the failed attempts
     * 
     * @param string username
     * @param string ip_address
     * @param integer ip_attempts
     */
    public function mo_website_security_add_ip($username, $ip_address, $ip_attempts)
    {
        $this->connection->update('miniorange_website_security_ip_track')
            ->fields(['login_attempts' => $ip_attempts])
            ->condition('ip_address', $ip_address, '=')
            ->execute();

        $this->miniorangeReportsRepository->mo_website_security_add_reports_entry($ip_address, $username, 'IP Login Failed');
    }

    /**
     * To block the given IP address
     * 
     * @param string username
     * @param string ip_address
     * @param string type
     */
    public function mo_website_security_block_ip($username, $ip_address, $type)
    {
        $this->connection->update('miniorange_website_security_ip_track')
            ->fields([
                'ip_type' => $type,
                'by_admin' => 0,
                'blocked_timestamp' => \Drupal::time()->getRequestTime(),
            ])
            ->condition('ip_address', $ip_address, '=')
            ->execute();

        $this->miniorangeReportsRepository->mo_website_security_add_reports_entry($ip_address, $username, 'IP Blocked');
    }

    /**
     * Returns the timestamp of the failed attempt to calculate the time difference between the 2 attempts
     * 
     * @param string ip_address
     */
    public function mo_website_security_get_timestamp_ip($ip_address)
    {
        $ip_timestamp = $this->connection->select('miniorange_website_security_ip_track', 'ns_ip')
            ->fields('ns_ip', ['failed_timestamp'])
            ->condition('ip_address', $ip_address)
            ->execute()
            ->fetchField();

        return $ip_timestamp;
    }

    /**
     * To check if current Client's IP is blacklisted
     */
    public function mo_website_security_is_ip_blacklisted()
    {
        $db_var = \Drupal::config('security_login_secure.settings');
        $block_ips = $db_var->get('website_security_block_ips');
        $block_ips = explode(';', $block_ips);
        $user_ip = \Drupal::request()->getClientIp();

        if (array_search($user_ip, $block_ips) === FALSE) {
            return FALSE;
        } else {
            if (\Drupal::currentUser()->isAuthenticated())
                user_logout();

            return TRUE;
        }
    }

    /**
     * To unblock the IP address
     * 
     * @param string ip_address
     * @param string uname
     */
    public function mo_website_security_unblock_ip($ip_address, $uname = "")
    {
        global $base_url;

        $this->connection->delete('miniorange_website_security_ip_track')
            ->condition('ip_address', $ip_address, '=')
            ->execute();

        $account = $this->entityTypeManager->getStorage('user')->load(\Drupal::currentUser()->id());
        $uname = $account->getAccountName();

        $this->miniorangeReportsRepository->mo_website_security_add_reports_entry($ip_address, $uname, 'Unblocked IP');
        $url = $base_url . '/admin/config/people/security_login_secure/WebsiteSecurityReports';
        $this->messenger->addStatus(t('The IP address ' . $ip_address . ' has been successfully unblocked.'));

        \Drupal::service('page_cache_kill_switch')->trigger();

        $response = new RedirectResponse($url);
        $response->send();
    }
}
