<?php

namespace Drupal\security_login_secure\Repository;

use Drupal\Core\Database\Connection;
use Drupal\security_login_secure\Repository\MiniorangeReportsRepository;
use Drupal\Core\Messenger\MessengerInterface;

class MiniorangeUserSecurityRepository
{
    /**
     * The database connection.
     *
     * @var Connection
     */
    protected $connection;

    /**
     * The Miniorange Reports Repository
     * 
     * @var MiniorangeReportsRepository
     */
    protected $miniorangeReportsRepository;

    /**
     * The messenger service.
     *
     * @var \Drupal\Core\Messenger\MessengerInterface
     */
    protected $messenger;

    /**
     * Construct a repository object.
     *
     * @param Connection $connection
     *   The database connection.
     * @param MiniorangeReportsRepository $miniorangeReportsRepository
     *   The Miniorange Reports Repository
     * @param \Drupal\Core\Messenger\MessengerInterface $messenger
     *   The messenger service.
     */
    public function __construct(Connection $connection, MiniorangeReportsRepository $miniorangeReportsRepository, MessengerInterface $messenger)
    {
        $this->connection = $connection;
        $this->miniorangeReportsRepository = $miniorangeReportsRepository;
        $this->messenger = $messenger;
    }

    /**
     * Flushes the Drupal's default table to override the blocking logic
     */
    public function mo_website_security_clear_flood_table_events_user()
    {
        if ($this->connection->schema()->tableExists('flood')) {
            try {
                $this->connection->delete('flood')
                    ->condition('event', 'user.failed_login_user', '=')
                    ->execute();
            } catch (\Exception $exception) {
                $this->messenger->addError(t('Something went wrong. Please see your recent log messages for details'));
                \Drupal::logger('security_login_secure.info')->info($exception);
            }
        }
    }

    /**
     * Unblock the user once the blocking time completes
     * 
     * @param string ip_address - IP address to unblock
     * @param string username 
     * @param object account - Account data of the user
     * @param boolean is_blocked - to unblock the user if blocked
     */
    public function mo_website_security_clear_events_users($ip_address, $username, $account, $is_blocked = true)
    {
        $deleted = 0;
        $db_var = \Drupal::config('security_login_secure.settings');
        $blocked_time_period = $db_var->get('website_security_user_block_time_period');

        if ($is_blocked == true) {

            if ($blocked_time_period > 0) {
                $blocked_time_period *= 3600;
                $time = \Drupal::time()->getRequestTime() - $blocked_time_period;

                if (!empty($account) && $account->status->value == 0) {
                    $deleted = $this->connection->delete('miniorange_website_security_user_track')
                        ->condition('blocked_timestamp', $time, '<')
                        ->condition('uname', $username, '=')
                        ->execute();

                    if ($deleted) {
                        $account->status->setValue(1);
                        $account->save();
                        $this->miniorangeReportsRepository->mo_website_security_add_reports_entry($ip_address, $username, 'Unblocked User');
                        website_security_delete_session_values();
                        return 1;
                    }
                }
            }
        } else {
            $deleted = $this->connection->delete('miniorange_website_security_user_track')
                ->condition('uname', $username, '=')
                ->execute();

            return 0;
        }
    }

    /**
     * Returns number of failed attempts from the given user
     * 
     * @param string username
     */
    public function mo_website_security_get_user_attempts($username)
    {
        $user_attempts = 0;
        $user_exists = $this->connection->select('miniorange_website_security_user_track', 'uname')
            ->fields('uname')
            ->condition('uname', $username)
            ->countQuery()
            ->execute()
            ->fetchField();

        if ($user_exists == 0) {
            $this->connection->insert('miniorange_website_security_user_track')
                ->fields([
                    'uname' => $username,
                    'failed_timestamp' => \Drupal::time()->getRequestTime(),
                ])
                ->execute();
        } else {
            $user_attempts = $this->connection->select('miniorange_website_security_user_track', 'ns_user')
                ->fields('ns_user', ['login_attempts'])
                ->condition('uname', $username)
                ->execute()
                ->fetchField();
        }

        return $user_attempts;
    }

    /**
     * To log the user in DB table to track the failed attempts
     * 
     * @param string username
     * @param string ip_address - IP Address of the current client
     * @param integer user_attempts - number of failed attempts
     */
    public function mo_website_security_add_user($username, $ip_address, $user_attempts)
    {
        $this->connection->update('miniorange_website_security_user_track')
            ->fields(['login_attempts' => $user_attempts])
            ->condition('uname', $username, '=')
            ->execute();

        $this->miniorangeReportsRepository->mo_website_security_add_reports_entry($ip_address, $username, 'User Login Failed');
    }

    /**
     * To block the user on exceeding failed attempts
     * 
     * @param string ip_address
     * @param string username
     */
    public function mo_website_security_block_user($ip_address, $username)
    {
        $account = user_load_by_name($username);

        if ($account->status->value == 1) {
            $account->status->setValue(0);
            $account->save();

            $this->connection->update('miniorange_website_security_user_track')
                ->fields([
                    'blocked_timestamp' => \Drupal::time()->getRequestTime(),
                ])
                ->condition('uname', $username, '=')
                ->execute();
        }

        $this->miniorangeReportsRepository->mo_website_security_add_reports_entry($ip_address, $username, 'User Blocked');
    }

    /**
     * Returns the timestamp on user marked as blocked
     * 
     * @param string username
     */
    public function mo_website_security_user_blocked_invalid_attempt($username)
    {
        $user_blocked = $this->connection->select('miniorange_website_security_user_track', 'ns_user')
            ->fields('ns_user', ['blocked_timestamp'])
            ->condition('uname', $username)
            ->execute()
            ->fetchField();

        return $user_blocked;
    }

    /**
     * Return timestamp of failed attempt to calculate the time difference between 2 attempts
     * 
     * @param string username
     */
    public function mo_website_security_get_timestamp_user($username)
    {
        $user_timestamp = $this->connection->select('miniorange_website_security_user_track', 'ns_user')
            ->fields('ns_user', ['failed_timestamp'])
            ->condition('uname', $username)
            ->execute()
            ->fetchField();

        return $user_timestamp;
    }

    /**
     * Flushes logs from user_track table on user login success
     */
    public function mo_website_security_delete_logs_on_login_success($account, $ip_address)
    {
        $deleted = $this->connection->delete('miniorange_website_security_user_track')
            ->condition('uname', $account->getAccountName(), '=')
            ->execute();

        $this->miniorangeReportsRepository->mo_website_security_add_reports_entry($ip_address, $account->getAccountName(), 'Success');
        website_security_delete_session_values();
    }

    /**
     * Flushes logs from user_track table on user update
     */
    public function mo_website_security_delete_logs_on_user_update($account)
    {
        $deleted = $this->connection->delete('miniorange_website_security_user_track')
            ->condition('uname', $account->getAccountName(), '=')
            ->execute();
        if ($deleted) {
            website_security_delete_session_values();
        }
    }
}
