<?php

namespace Drupal\seeds_layouts\Plugin;

use Drupal\Component\Utility\NestedArray;
use Drupal\Component\Utility\Xss;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManager;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Layout\LayoutDefault;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\seeds_layouts\SeedsLayoutsManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Layout class for all Display Suite layouts.
 */
class SeedsLayout extends LayoutDefault implements PluginFormInterface, ContainerFactoryPluginInterface {

  /**
   * Defines if the current user uses user friendly layout.
   *
   * @var bool
   */
  protected $userFriendly;

  /**
   * The entity type manager interface.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Configuration of seeds layouts.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $seedsLayoutConfig;

  /**
   * Configuration of seeds layouts styles.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $seedsColumnsConfig;

  /**
   * Configuration of seeds layouts Editable .
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $seedsLayoutEditableConfig;

  /**
   * Seeds layouts manager.
   *
   * @var \Drupal\seeds_layouts\SeedsLayoutsManager
   */
  protected $seedsLayoutsManager;

  /**
   * Session account proxy interface.
   *
   * @var Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Layout field manager.
   *
   * @var LayoutFieldManager
   */
  protected $layoutFieldManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ConfigFactoryInterface $config_factory, SeedsLayoutsManager $seeds_layouts_manager, LayoutFieldManager $layout_field_manager, AccountProxyInterface $current_user, EntityTypeManager $entity_type_manager) {

    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->seedsLayoutConfig = $config_factory->get('seeds_layouts.config');
    $this->seedsLayoutEditableConfig = $config_factory->getEditable('seeds_layouts.config');
    $this->seedsColumnsConfig = $config_factory->get('seeds_layouts.columns');
    $this->seedsLayoutsManager = $seeds_layouts_manager;
    $this->layoutFieldManager = $layout_field_manager;
    $this->currentUser = $current_user;
    $this->userFriendly = !$current_user->hasPermission("access advanced seeds layouts settings");
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritDoc}.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('seeds_layouts.manager'),
      $container->get('plugin.manager.layout_field'),
      $container->get('current_user'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Initialization.
    $configuration = $this->getConfiguration();

    if (!isset($configuration["seeds_container"]["seeds_body"]["fields"]['layout_fields'])) {
      $configuration["seeds_container"]["seeds_body"]["fields"]['layout_fields'] = [];

    }

    $form['seeds_container'] = [
      '#type' => 'container',
      '#weight' => -50,
      '#attributes' => [
        'id' => 'seeds-container',
      ],
    ];

    $tabs = [
          [
            'id' => 'layout',
            'icon' => 'layout.svg',
            'title' => $this->t('Layout'),
            'active' => TRUE,
          ],
          [
            'id' => 'fields',
            'icon' => 'field.svg',
            'title' => $this->t('Field'),
          ],
          [
            'id' => 'advanced',
            'icon' => 'advanced.svg',
            'title' => $this->t('Advanced'),
          ],
    ];

    $form['seeds_container']['seeds_tabs'] = [
      '#type' => 'html_tag',
      '#tag' => 'ul',
      '#attributes' => [
        'class' => 'seeds-tabs',
        'role' => 'tablist',
      ],
    ];

    $form['seeds_container']['seeds_body'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => 'seeds-body',
        'id' => 'seeds-body',
      ],
    ];

    foreach ($tabs as $tab) {
      $form['seeds_container']['seeds_tabs'][$tab['id']] = [
        '#type' => 'inline_template',
        '#template' => '<li><a data-target="{{ target|clean_class }}" class="{{active}}">{% include icon %}<p class="title">{{ title }}</p></a></li>',
        '#context' => [
          'title' => $tab['title'],
          'target' => $tab['id'],
          'active' => isset($tab['active']) && $tab['active'] == TRUE ? 'active' : '',
          'icon' => \Drupal::service('extension.list.module')->getPath('seeds_layouts') . '/assets/images/' . $tab['icon'],
        ],
      ];

      $form['seeds_container']['seeds_body'][$tab['id']] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => [
            'seeds-body--' . $tab['id'],
            'seeds-body-item',
            isset($tab['active']) && $tab['active'] == TRUE ? 'active' : '',
          ],
        ],
      ];
    }

    $regions = $this->getPluginDefinition()->getRegions();
    $regions_count = count($regions);
    $sizes = [
      'desktop' => $this->t("Desktop"),
      'tablet' => $this->t("Tablet"),
      'mobile' => $this->t("Mobile"),
    ];

    // Custom Fields.
    $layout_fields = $this->seedsLayoutConfig->get('layout_fields');
    /* @todo Add sortable drag and drop table */
    $form['seeds_container']['seeds_body']['fields']['layout_fields'] = [
      '#type' => 'container',
      '#tree' => TRUE,
    ];
    foreach ($layout_fields as $key => $layout_field) {
      $config = $layout_field['custom_field'];
      $description = $config['description'];
      // Ignore _none plugins if exist.
      if ($description['type'] == '_none') {
        continue;
      }
      // Create custom fields instances and build their forms.
      $default_values = @$configuration["seeds_container"]["seeds_body"]["fields"]['layout_fields'][$key];
      /** @var \Drupal\seeds_layouts\Plugin\LayoutFieldInterface $field_instance */
      $field_instance = $this->layoutFieldManager->createInstance($description['type']);
      // Set the uuid and the layout.
      $field_instance->setUuid($key);
      $field_instance->setLayout($this->getPluginDefinition());
      // Set configurations.
      $default_config = $field_instance->defaultConfiguration();
      $field_instance->setConfiguration(NestedArray::mergeDeepArray([$default_config, $config, $default_values]));
      // Initialize an empty div subform.
      $empty_form = ['#type' => 'container'];
      $subform_state = SubformState::createForSubform($empty_form, $form, $form_state);

      // Call the 'build' method from the 'LayoutField' instance.
      $field_form = $field_instance->build($empty_form, $subform_state);
      // Set the form weight.
      $field_form['#weight'] = $layout_field['weight'];

      // @todo Determine if this line is important.
      $field_form['type'] = [
        '#type' => 'value',
        '#value' => $description['type'],
      ];

      // Render the custom fields.
      $form['seeds_container']['seeds_body']['fields']['layout_fields'][$key] = $field_form;
    }

    // Define the three select sizes.
    $mobile = $tablet = $desktop = [
      '#type' => 'radios',
      '#options' => [],
      '#access' => FALSE,
    ];

    $columns = $this->seedsColumnsConfig->get();
    foreach ($columns as $column) {
      // Check if this column attribute belongs to this region.
      if (!is_array($column) || $column['description']['type'] != $this->getPluginDefinition()->getThemeHook()) {
        continue;
      }

      // Get the column info.
      $size = $column['description']['size'];
      $label = $column['description']['label'];
      $classes = implode(',', $column['classes']);
      $image = $column['image_url'];

      // Or simply add it to its appropiate size element.
      ${$size}['#options'][$classes] = '<img class="layout-icon" src="' . $image . '" />';
    }

    // Loop through the $sizes and build them.
    foreach ($sizes as $size => $label) {
      ${$size}['#title'] = $label;
      ${$size}['#default_value'] = $configuration["seeds_container"]["seeds_body"]["layout"]["columns"][$size];

      if (${$size}['#options']) {
        // If the size has options, we reveal it.
        ${$size}['#access'] = TRUE;
      }
    }

    // A container to wrap the size elements.
    $form['seeds_container']['seeds_body']['layout']['columns'] = [
      '#type' => 'container',
      '#tree' => TRUE,
      'desktop' => $desktop,
      'tablet' => $tablet,
      'mobile' => $mobile,
    ];

    // A checkbox to reverse the order of the columns.
    if ($regions_count > 1) {
      $form['seeds_container']['seeds_body']['layout']['reverse'] = [
        '#type' => 'checkbox',
        '#title' => $this->t("Reverse"),
        '#default_value' => $configuration["seeds_container"]["seeds_body"]["layout"]['reverse'],
      ];
    }

    // Container checkbox (If the current framework supports containers).
    if ($this->seedsLayoutConfig->get('framework_has_container')) {
      $form['seeds_container']['seeds_body']['layout']['container'] = [
        '#type' => 'checkbox',
        // We simply flip the title and values if the current user
        // doesn't user advanced layouts.
        '#title' => $this->userFriendly ? t("Full Width") : t("Contianer"),
        '#default_value' => $this->userFriendly ? !$configuration["seeds_container"]["seeds_body"]["layout"]['container'] : $configuration["seeds_container"]["seeds_body"]["layout"]['container'],
      ];
    }

    // The region attributes field.
    foreach ($regions as $id => $region) {
      $region_saved_attributes = $configuration["seeds_container"]["seeds_body"]["advanced"][$id . '_attributes'] ?? "";
      $region_default_attributes = $regions_count == 1 ? $this->seedsLayoutConfig->get('one_column_attributes') : "";
      $form['seeds_container']['seeds_body']['advanced'][$id . "_attributes"] = [
        '#type' => 'textfield',
        '#title' => $this->t("%region Attributes", ['%region' => $region['label']]),
        '#default_value' => $region_saved_attributes ? $region_saved_attributes : $region_default_attributes,
      ];
    }

    // The section attributes field.
    $form['seeds_container']['seeds_body']['advanced']['section_attributes'] = [
      '#type' => 'textfield',
      '#title' => $this->t("%region Attributes", ['%region' => $this->t("Section")]),
      '#default_value' => $configuration["seeds_container"]["seeds_body"]["advanced"]['section_attributes'],
    ];

    $override_columns_parent_attributes = $configuration["seeds_container"]["seeds_body"]["advanced"]['columns_parent_attributes'];
    $form['seeds_container']['seeds_body']['advanced']['columns_parent_attributes'] = [
      '#type' => 'textfield',
      '#title' => $this->t("%region Attributes", ['%region' => $this->t("Column Parents")]),
      '#default_value' => $override_columns_parent_attributes ? $override_columns_parent_attributes : $this->seedsLayoutConfig->get('columns_parent_attributes'),
      '#description' => $this->t('Used as: @example', ['@example' => '"class|example-class,data-example|example-value"']),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    $regions = $this->getPluginDefinition()->getRegions();

    // Set columns width.
    $columns = $values['seeds_container']['seeds_body']['layout']['columns'];
    // Loop through the sizes elements.
    foreach ($regions as $region => $label) {
      foreach ($columns as $column) {
        if (isset($column) && $column != "_none") {
          $classes = $this->seedsLayoutsManager->sortClassesPerRegion($column, $regions);
          $values['seeds_container']['seeds_body']['layout'][$region . '_column'][] = $classes[$region];
        }
      }
    }

    // Set advanced attributes.
    foreach ($regions as $key => $name) {
      $values["seeds_container"]["seeds_body"]["advanced"][$key . "_attributes"] = Xss::filter($values["seeds_container"]["seeds_body"]['advanced'][$key . "_attributes"]);
    }
    $values["seeds_container"]["seeds_body"]["advanced"]['section_attributes'] = Xss::filter($values["seeds_container"]["seeds_body"]['advanced']["section_attributes"]);
    $values["seeds_container"]["seeds_body"]["advanced"]['columns_parent_attributes'] = Xss::filter($values["seeds_container"]["seeds_body"]['advanced']["columns_parent_attributes"]);

    // Get attributes from custom fields.
    $field_config = $this->seedsLayoutConfig->get('layout_fields');
    $layout_fields_attributes = [];

    if (isset($values["seeds_container"]["seeds_body"]["fields"]['layout_fields'])) {
      foreach ($values["seeds_container"]["seeds_body"]["fields"]['layout_fields'] as $key => $config) {
        $field_instance = $this->layoutFieldManager->createInstance($config['type'], NestedArray::mergeDeepArray([$config, $field_config[$key]['custom_field']]));
        $field_attributes = $field_instance->getAttributes();
        $layout_fields_attributes = NestedArray::mergeDeepArray([$field_attributes, $layout_fields_attributes]);
      }
    }

    // Finalization.
    $values["seeds_container"]["seeds_body"]["fields"]['layout_fields_attributes'] = $layout_fields_attributes;
    $values["seeds_container"]["seeds_body"]["layout"]['container'] = $this->userFriendly ? !$values["seeds_container"]["seeds_body"]["layout"]['container'] : $values["seeds_container"]["seeds_body"]["layout"]['container'];
    $this->setConfiguration($values);
  }

}
