#!/bin/bash
# Script to quickly create a sub-theme.

# Function to display error messages.
displayError() {
    echo -e "\033[41m$1\033[0m\033[K"
}

# Function to exit if an error occurs.
exitIfError() {
    if [ $1 -ne 0 ]; then
        displayError "Error: $2"
        exit 1
    fi
}

# Function to find Drupal root directory
findDrupalRoot() {
    local current_dir="$1"
    local max_depth=10
    local depth=0
    
    while [ "$depth" -lt "$max_depth" ] && [ "$current_dir" != "/" ]; do
        # Check for common Drupal directory structures
        if [ -f "$current_dir/composer.json" ]; then
            # Check for web/sites/default/settings.php (standard Drupal structure)
            if [ -f "$current_dir/web/sites/default/settings.php" ]; then
                echo "$current_dir"
                return 0
            fi
            # Check for public_html/sites/default/settings.php (your structure)
            if [ -f "$current_dir/public_html/sites/default/settings.php" ]; then
                echo "$current_dir"
                return 0
            fi
            # Check for sites/default/settings.php (direct structure)
            if [ -f "$current_dir/sites/default/settings.php" ]; then
                echo "$current_dir"
                return 0
            fi
        fi
        current_dir="$(dirname "$current_dir")"
        depth=$((depth + 1))
    done
    
    return 1
}

# Function to find Drush executable
findDrush() {
    local drupal_root="$1"
    
    # Check for Drush in common locations
    local drush_paths=(
        "$drupal_root/vendor/bin/drush"
        "$drupal_root/bin/drush"
        "$drupal_root/web/vendor/bin/drush"
        "$drupal_root/web/bin/drush"
        "$drupal_root/public_html/vendor/bin/drush"
        "$drupal_root/public_html/bin/drush"
        "$drupal_root/../vendor/bin/drush"
        "$drupal_root/../bin/drush"
    )
    
    for path in "${drush_paths[@]}"; do
        if [ -x "$path" ]; then
            echo "$path"
            return 0
        fi
    done
    
    # Try to find drush in PATH
    if command -v drush >/dev/null 2>&1; then
        echo "drush"
        return 0
    fi
    
    return 1
}

# Display script information.
echo '
+------------------------------------------------------------------------+
| Use this script to quickly create a seeds_ui sub-theme.               |
| Instructions:                                                         |
| - The seeds_ui theme folder must be in the contrib folder.            |
+------------------------------------------------------------------------+
'

# Variables
SUBTHEME_SOURCE="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )/../subtheme"

# Find Drupal root first
echo "Finding Drupal installation..."
DRUPAL_ROOT=$(findDrupalRoot "$(pwd)")
if [ $? -ne 0 ]; then
    displayError "Could not find Drupal root directory. Please ensure you're running this script from within a Drupal project."
    exit 1
fi

echo "Found Drupal root: $DRUPAL_ROOT"

# Get theme information.
read -p 'Enter the machine name of your custom theme (e.g., hello_world): ' CUSTOM_SEEDS_THEME
if [ -z "$CUSTOM_SEEDS_THEME" ]; then
    displayError "Custom theme machine name cannot be empty."
    exit 1
fi

read -p 'Enter your theme name (e.g., Hello World): ' CUSTOM_SEEDS_THEME_NAME
if [ -z "$CUSTOM_SEEDS_THEME_NAME" ]; then
    displayError "Theme name cannot be empty."
    exit 1
fi

# Let user specify themes directory relative to Drupal root
echo "Drupal root found at: $DRUPAL_ROOT"

read -p 'Enter the themes directory path (relative to Drupal root, e.g., web/themes): ' THEMES_RELATIVE_PATH
if [ -z "$THEMES_RELATIVE_PATH" ]; then
    displayError "Themes directory path cannot be empty."
    exit 1
fi

THEMES_DIRECTORY="$DRUPAL_ROOT/$THEMES_RELATIVE_PATH"

# Create the directory if it doesn't exist
if [ ! -d "$THEMES_DIRECTORY" ]; then
    echo "Creating themes directory at: $THEMES_DIRECTORY"
    mkdir -p "$THEMES_DIRECTORY"
fi

echo "Using themes directory: $THEMES_DIRECTORY"

# Create sub-theme directory.
DESTINATION_PATH="$THEMES_DIRECTORY/$CUSTOM_SEEDS_THEME"
if [ -d "$DESTINATION_PATH" ]; then
    displayError "The directory $DESTINATION_PATH already exists. Please choose a different name."
    exit 1
fi

echo "Copying sub-theme to $DESTINATION_PATH..."
cp -r "$SUBTHEME_SOURCE" "$DESTINATION_PATH"
exitIfError $? "Failed to copy the sub-theme to $DESTINATION_PATH."

cd "$DESTINATION_PATH" || exitIfError $? "Failed to navigate to $DESTINATION_PATH."

# Rename files containing "THEMENAME".
echo "Renaming theme files..."
for file in *THEMENAME.*; do mv "$file" "${file//THEMENAME/$CUSTOM_SEEDS_THEME}"; done
for file in config/*/*THEMENAME*; do mv "$file" "${file//THEMENAME/$CUSTOM_SEEDS_THEME}"; done

# Replace occurrences of "THEMENAME" in files.
echo "Replacing placeholders in theme files..."
grep -Rl . . | xargs sed -i "s/THEMENAME/$CUSTOM_SEEDS_THEME/g"
exitIfError $? "Failed to replace placeholders in theme files."

# Install dependencies and build theme assets.
echo "Installing dependencies and building theme assets..."
npm install &>/dev/null
exitIfError $? "npm install failed."

npm run css &>/dev/null
exitIfError $? "npm run css failed."

# Find Drush executable
echo "Finding Drush executable..."
DRUSH_PATH=$(findDrush "$DRUPAL_ROOT")
if [ $? -ne 0 ]; then
    displayError "Could not find Drush executable. Please ensure Drush is installed."
    exit 1
fi

echo "Found Drush: $DRUSH_PATH"

# Enable theme in Drupal and set as default.
echo "Enabling and setting the theme as default in Drupal..."
cd "$DRUPAL_ROOT" || exitIfError $? "Failed to navigate to Drupal root."

"$DRUSH_PATH" theme:enable "$CUSTOM_SEEDS_THEME" -y
exitIfError $? "Failed to enable the theme."

"$DRUSH_PATH" config-set system.theme default "$CUSTOM_SEEDS_THEME" -y
exitIfError $? "Failed to set the theme as default."

echo -e "\033[42mSub-theme $CUSTOM_SEEDS_THEME successfully created and enabled!\033[0m"
