<?php

namespace Drupal\seeds_widgets\Plugin\Field\FieldFormatter;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceFormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\seeds_widgets\SeedsWidgetsManager;
use Drupal\seeds_widgets\SeedsWidgetTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'seeds_widgets_paragraph' formatter.
 *
 * @FieldFormatter(
 *   id = "seeds_widgets_paragraph",
 *   label = @Translation("Seeds Widgets Paragraph"),
 *   field_types = {
 *     "entity_reference_revisions"
 *   }
 * )
 */
class SeedsWidgetsParagraphFormatter extends EntityReferenceFormatterBase implements ContainerFactoryPluginInterface, SeedsWidgetTypeInterface {

  /**
   * The Seeds Widgets Manager.
   *
   * @var \Drupal\seeds_widgets\SeedsWidgetsManager
   */
  protected $seedsWidgetsManager;

  /**
   * Constructs a SeedsWidgetsParagraphFormatter object.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\seeds_widgets\SeedsWidgetsManager $seeds_widgets_manager
   *   The Seeds Widgets Manager.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, SeedsWidgetsManager $seeds_widgets_manager) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->seedsWidgetsManager = $seeds_widgets_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('seeds_widgets.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function getSupportedWidgetTypes(): array {
    return [
      'vertical_tabs' => new TranslatableMarkup('Vertical Tabs'),
      'horizontal_tabs' => new TranslatableMarkup('Horizontal Tabs'),
      'modal' => new TranslatableMarkup('Modal'),
      'accordion' => new TranslatableMarkup('Accordion'),
      'grid' => new TranslatableMarkup('Grid'),
      'slider' => new TranslatableMarkup('Slider'),
      'carousel' => new TranslatableMarkup('Carousel'),
      'timeline' => new TranslatableMarkup('Timeline'),
      'default' => new TranslatableMarkup('Default'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'widget_type' => 'default',
      'grid_desktop' => 'col-lg-4',
      'grid_tablet' => 'col-md-6',
      'grid_mobile' => 'col-12',
      'modal_title' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);

    $form['widget_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Widget Type'),
      '#options' => $this->getSupportedWidgetTypes(),
      '#default_value' => $this->getSetting('widget_type'),
      '#description' => $this->t('Select the type of widget to render the paragraphs.'),
    ];

    // Grid-specific settings.
    $form['grid_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Grid Settings'),
      '#states' => [
        'visible' => [
          ':input[name="fields[field_seeds_widget_paragraph][settings_edit_form][settings][widget_type]"]' => ['value' => 'grid'],
        ],
      ],
    ];

    $form['grid_settings']['grid_desktop'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Desktop CSS classes'),
      '#default_value' => $this->getSetting('grid_desktop'),
      '#description' => $this->t('CSS classes for desktop grid items (e.g., col-lg-4).'),
    ];

    $form['grid_settings']['grid_tablet'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tablet CSS classes'),
      '#default_value' => $this->getSetting('grid_tablet'),
      '#description' => $this->t('CSS classes for tablet grid items (e.g., col-md-6).'),
    ];

    $form['grid_settings']['grid_mobile'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Mobile CSS classes'),
      '#default_value' => $this->getSetting('grid_mobile'),
      '#description' => $this->t('CSS classes for mobile grid items (e.g., col-12).'),
    ];

    // Modal-specific settings.
    $form['modal_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Modal Settings'),
      '#states' => [
        'visible' => [
          ':input[name="fields[field_seeds_widget_paragraph][settings_edit_form][settings][widget_type]"]' => ['value' => 'modal'],
        ],
      ],
    ];

    $form['modal_settings']['modal_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Modal Title'),
      '#default_value' => $this->getSetting('modal_title'),
      '#description' => $this->t('Title for the modal dialog.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();

    $widget_type = $this->getSetting('widget_type');
    $summary[] = $this->t('Widget type: @type', ['@type' => $widget_type]);

    if ($widget_type === 'grid') {
      $summary[] = $this->t('Grid classes: @desktop, @tablet, @mobile', [
        '@desktop' => $this->getSetting('grid_desktop'),
        '@tablet' => $this->getSetting('grid_tablet'),
        '@mobile' => $this->getSetting('grid_mobile'),
      ]);
    }

    if ($widget_type === 'modal' && $modal_title = $this->getSetting('modal_title')) {
      $summary[] = $this->t('Modal title: @title', ['@title' => $modal_title]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $widget_type = $this->getSetting('widget_type');

    if ($items->isEmpty()) {
      return $elements;
    }
    // Get the entities and build the render array.
    $entities = $this->getEntitiesToView($items, $langcode);
    $processed_items = [];

    foreach ($entities as $delta => $entity) {
      $view_builder = \Drupal::entityTypeManager()->getViewBuilder($entity->getEntityTypeId());

      // Check if the paragraph has a custom view mode field and use that.
      $view_mode = $this->viewMode;
      if ($entity instanceof ContentEntityInterface &&
          $entity->hasField('field_seeds_widget_view_mode') &&
          !$entity->get('field_seeds_widget_view_mode')->isEmpty()) {
        $custom_view_mode = $entity->get('field_seeds_widget_view_mode')->target_id;
        if ($custom_view_mode) {
          $view_mode = str_replace("paragraph.", "", $custom_view_mode);
        }
      }

      $processed_items[$delta] = [
        'content' => $view_builder->view($entity, $view_mode, $entity->language()->getId()),
        '#paragraph' => $entity,
      ];
    }

    // Build the base render array with widget-specific theme.
    $template_name = 'seeds_widgets_paragraph_formatter__' . str_replace('-', '_', $widget_type);
    $elements[0] = [
      '#theme' => $template_name,
      '#items' => $processed_items,
      '#widget_type' => $widget_type,
      '#settings' => $this->getSettings(),
    ];
    return $elements;
  }

}
