<?php

namespace Drupal\seeds_widgets\Plugin\Field\FieldFormatter;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceFormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\seeds_widgets\SeedsWidgetsManager;
use Drupal\seeds_widgets\SeedsWidgetTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'timeline' formatter.
 *
 * @FieldFormatter(
 *   id = "seeds_widgets_timeline",
 *   label = @Translation("Seeds Widgets Timeline"),
 *   field_types = {
 *     "entity_reference_revisions"
 *   }
 * )
 */
class TimelineFormatter extends EntityReferenceFormatterBase implements ContainerFactoryPluginInterface, SeedsWidgetTypeInterface {

  /**
   * The Seeds Widgets Manager.
   *
   * @var \Drupal\seeds_widgets\SeedsWidgetsManager
   */
  protected $seedsWidgetsManager;

  /**
   * Constructs a TimelineFormatter object.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\seeds_widgets\SeedsWidgetsManager $seeds_widgets_manager
   *   The Seeds Widgets Manager.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, SeedsWidgetsManager $seeds_widgets_manager) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->seedsWidgetsManager = $seeds_widgets_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('seeds_widgets.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'alternate_sides' => TRUE,
      'animate' => TRUE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);
    $settings = $this->getSettings();

    $form['alternate_sides'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Alternate Sides'),
      '#default_value' => $settings['alternate_sides'],
      '#description' => $this->t('Alternate content between left and right sides.'),
    ];

    $form['animate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Animate'),
      '#default_value' => $settings['animate'],
      '#description' => $this->t('Add animation effects when timeline items come into view.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();
    $settings = $this->getSettings();

    if ($settings['alternate_sides']) {
      $summary[] = $this->t('Alternating sides');
    }

    if ($settings['animate']) {
      $summary[] = $this->t('With animation');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    if ($items->isEmpty()) {
      return $elements;
    }

    $settings = $this->getSettings();
    $entities = $this->getEntitiesToView($items, $langcode);
    $processed_items = [];

    foreach ($entities as $delta => $entity) {
      if (!$entity) {
        continue;
      }

      $view_builder = \Drupal::entityTypeManager()->getViewBuilder($entity->getEntityTypeId());

      // Check if the paragraph has a custom view mode field and use that.
      $view_mode = $this->viewMode;
      if ($entity instanceof ContentEntityInterface &&
          $entity->hasField('field_seeds_widget_view_mode') &&
          !$entity->get('field_seeds_widget_view_mode')->isEmpty()) {
        $custom_view_mode = $entity->get('field_seeds_widget_view_mode')->target_id;
        if ($custom_view_mode) {
          $view_mode = str_replace("paragraph.", "", $custom_view_mode);
        }
      }

      $processed_items[$delta] = [
        'content' => $view_builder->view($entity, $view_mode, $entity->language()->getId()),
        '#paragraph' => $entity,
        'position' => $settings['alternate_sides'] ? ($delta % 2 ? 'right' : 'left') : 'left',
      ];
    }

    // Build the base render array with widget-specific theme.
    $elements[0] = [
      '#theme' => 'seeds_widgets_paragraph_formatter__timeline',
      '#items' => $processed_items,
      '#widget_type' => 'timeline',
      '#settings' => $settings,
      '#attached' => [
        'library' => ['seeds_widgets/timeline'],
      ],
    ];

    return $elements;
  }

}
