/**
 * Seeds Widgets Tabs Functionality
 * Vanilla JavaScript implementation for tabs functionality
 */

(function (Drupal, once) {
  Drupal.behaviors.tabsBehavior = {
    attach: function (context) {

      const TABS_SELECTOR = '.seeds-ui-tabs'

      function handleTabsStateCustom(tabContainer, tabButton) {
        // Get the tab index from data attribute
        const tabIndex = tabButton.getAttribute('data-tab')
        if (!tabIndex) return
        
        // Remove active class from all tab buttons
        const allButtons = tabContainer.querySelectorAll('.seeds-ui-tab-button')
        allButtons.forEach(button => {
          button.classList.remove('active')
          button.setAttribute('aria-selected', 'false')
          button.setAttribute('tabindex', '-1')
        })
        
        // Add active class to clicked button
        tabButton.classList.add('active')
        tabButton.setAttribute('aria-selected', 'true')
        tabButton.setAttribute('tabindex', '0')
        
        // Hide all content panels
        const allContent = tabContainer.querySelectorAll('.seeds-ui-tab-content')
        allContent.forEach(content => {
          content.classList.remove('active')
          content.setAttribute('aria-hidden', 'true')
        })
        
        // Show the corresponding content panel
        const activeContent = tabContainer.querySelector(`[data-tab="${tabIndex}"].seeds-ui-tab-content`)
        if (activeContent) {
          activeContent.classList.add('active')
          activeContent.setAttribute('aria-hidden', 'false')
        }
      }
      function setupTabs() {
        const tabsContainers = document.querySelectorAll(TABS_SELECTOR)
        
        tabsContainers.forEach(tabContainer => {
          const tabButtons = tabContainer.querySelectorAll('.seeds-ui-tab-button')
          const tabContent = tabContainer.querySelectorAll('.seeds-ui-tab-content')
          const tabsNav = tabContainer.querySelector('.seeds-ui-tabs-nav')
          
          // Initialize ARIA attributes
          if (tabsNav) {
            tabsNav.setAttribute('role', 'tablist')
            
            // Get all button IDs for aria-owns
            const buttonIds = Array.from(tabButtons).map((button, index) => {
              const existingId = button.getAttribute('id')
              if (!existingId) {
                const newId = `tab-${index + 1}`
                button.setAttribute('id', newId)
                return newId
              }
              return existingId
            })
            
            tabsNav.setAttribute('aria-owns', buttonIds.join(' '))
          }
          
          // Initialize tab buttons
          tabButtons.forEach((button, index) => {
            const buttonId = button.getAttribute('id') || `tab-${index + 1}`
            
            button.setAttribute('role', 'tab')
            button.setAttribute('aria-selected', 'false')
            button.setAttribute('aria-controls', `panel-${buttonId}`)
            button.setAttribute('tabindex', '-1')
            
            // Add click event listener
            button.addEventListener('click', (e) => {
              const clickedButton = e.target;
              handleTabsStateCustom(tabContainer, clickedButton)
            })
          })
          
          // Initialize content panels
          tabContent.forEach((content, index) => {
            const correspondingButton = tabButtons[index]
            const buttonId = correspondingButton?.getAttribute('id') || `tab-${index + 1}`
            
            content.setAttribute('role', 'tabpanel')
            content.setAttribute('id', `panel-${buttonId}`)
            content.setAttribute('aria-labelledby', buttonId)
            content.setAttribute('aria-hidden', 'true')
          })
          
          // Set initial active state
          const defaultActive = tabContainer.querySelector('.seeds-ui-tab-button.active') || tabButtons[0]
          if (defaultActive instanceof HTMLElement) {
            defaultActive.classList.add('active')
            defaultActive.setAttribute('tabindex', '0')
            defaultActive.setAttribute('aria-selected', 'true')
            
            // Show corresponding content
            const tabIndex = defaultActive.getAttribute('data-tab') || '1'
            const activeContent = tabContainer.querySelector(`[data-tab="${tabIndex}"].seeds-ui-tab-content`)
            if (activeContent) {
              activeContent.classList.add('active')
              activeContent.setAttribute('aria-hidden', 'false')
            }
          }
        })
        
        // Handle keyboard navigation
        const dir = getComputedStyle(document.body).getPropertyValue('direction')
        
        tabsContainers.forEach(tabContainer => {
          const tabs = tabContainer.querySelectorAll('.seeds-ui-tab-button')
          
          tabs.forEach(button => {
            if (button instanceof HTMLElement) {
              button.addEventListener('keydown', (e) => {
                switch (e.key) {
                  case 'ArrowRight':
                    e.preventDefault()
                    if (dir === 'ltr') {
                      navigateNextSibling(button, tabs)
                    } else {
                      navigatePrevSibling(button, tabs)
                    }
                    break
                    
                  case 'ArrowLeft':
                    e.preventDefault()
                    if (dir === 'ltr') {
                      navigatePrevSibling(button, tabs)
                    } else {
                      navigateNextSibling(button, tabs)
                    }
                    break
                    
                  case 'Tab':
                    // Reset tabindex for non-selected tabs
                    tabs.forEach(tab => {
                      if (tab.getAttribute('aria-selected') === 'false') {
                        tab.setAttribute('tabindex', '-1')
                      }
                    })
                    break
                    
                  case 'Enter':
                  case ' ':
                    e.preventDefault()
                    handleTabsStateCustom(tabContainer, button)
                    break
                }
              })
            }
          })
        })
      }
      setupTabs()
    },  
  };
})(Drupal, once);