/**
 * Seeds Widgets Accordion Functionality
 * Vanilla JavaScript implementation for accordion functionality
 */

(function (Drupal, once) {
  Drupal.behaviors.accordionBehavior = {
    attach: function (context) {
      function closeAccordion(item, content, header) {
        if (content instanceof HTMLElement) {
          content.setAttribute("aria-hidden", "true");
          header?.setAttribute("aria-expanded", "false");
          
          // Set current height first, then animate to 0
          content.style.maxHeight = content.scrollHeight + "px";
          
          // Force reflow to ensure the height is set
          content.offsetHeight;
          
          setTimeout(() => {
            content.style.maxHeight = "0";
            item.classList.remove("active");
          }, 10);
        } else {
          item.classList.remove("active");
        }
      }
      
      function setupAccordion() {
        // Use once() to ensure setupAccordion runs only once per accordion
        once('accordion-setup', '.seeds-ui-accordion', context).forEach((accordion) => {
          const accordionItems = accordion.querySelectorAll(
            ".seeds-ui-accordion-item"
          );
          
          accordionItems.forEach((item) => {
            accordion.setAttribute("role", "tabpanel");
            const key = item.getAttribute("data-key");
            const header = item.querySelector(".seeds-ui-accordion-header");
            const content = item.querySelector(".seeds-ui-accordion-content");
      
            if (key) {
              header?.setAttribute("id", "trigger-" + key);
              header?.setAttribute("aria-controls", key);
              header?.setAttribute("tabindex", "0");
      
              content?.setAttribute("id", key);
              content?.setAttribute("role", "region");
              content?.setAttribute("aria-labelledby", "trigger-" + key);
            }
      
            header?.setAttribute("aria-expanded", "false");

            // Set up transitionend listener once during initialization
            if (content instanceof HTMLElement) {
              content.addEventListener("transitionend", (e) => {
                const isActive = item?.classList.contains("active");
                if (isActive) {
                  content.style.maxHeight = "fit-content";
                } else {
                  content?.removeAttribute("style");
                }
              });
            }
      
            header?.addEventListener("click", (e) => {
              const isDisabled = item.classList.contains("disabled");
              const isActive = item?.classList.contains("active");
              
              console.log(isDisabled, 'isDisabled');
              console.log(isActive, 'isActive');
              
              if (isDisabled) return;
              
              if (isActive && content) {
                closeAccordion(item, content, header);
              } else {
                // Close open accordions before opening the new accordion
                accordionItems.forEach((accordionItem) => {
                  if (accordionItem.classList.contains("active")) {
                    const accordionHeader = accordionItem.querySelector(
                      ".seeds-ui-accordion-header"
                    );
                    const accordionContent = accordionItem.querySelector(
                      ".seeds-ui-accordion-content"
                    );
                    if (accordionContent && accordionHeader) {
                      closeAccordion(accordionItem, accordionContent, accordionHeader);
                    }
                  }
                });
      
                // Wait a bit for closing animations to start, then open the new one
                setTimeout(() => {
                  item.classList.add("active");
                  if (content instanceof HTMLElement) {
                    content.setAttribute("aria-hidden", "false");
                    header?.setAttribute("aria-expanded", "true");
                    
                    // Reset max-height to auto temporarily to get accurate scrollHeight
                    content.style.maxHeight = "auto";
                    const height = content.scrollHeight;
                    content.style.maxHeight = "0";
                    
                    // Force reflow
                    content.offsetHeight;
                    
                    // Animate to the correct height
                    setTimeout(() => {
                      content.style.maxHeight = height + "px";
                    }, 10);
                  }
                }, 50);
              }
            });
          });
        });
      }
      
      setupAccordion();
    },
  };
})(Drupal, once);