<?php

declare(strict_types=1);

namespace Drupal\seeds_widgets\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Block\BlockManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\views\Plugin\Block\ViewsBlock;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a blocks tabs block.
 *
 * @Block(
 *   id = "blocks_tabs",
 *   admin_label = @Translation("Blocks Tabs"),
 *   category = @Translation("Seeds Widgets"),
 * )
 */
final class BlocksTabsBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The block manager.
   *
   * @var \Drupal\Core\Block\BlockManagerInterface
   */
  protected $blockManager;

  /**
   * The context repository.
   *
   * @var \Drupal\Core\Plugin\Context\ContextRepositoryInterface
   */
  protected $contextRepository;

  /**
   * The UUID service.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected $uuidService;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, BlockManagerInterface $block_manager, ContextRepositoryInterface $context_repository, UuidInterface $uuid_service) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->blockManager = $block_manager;
    $this->contextRepository = $context_repository;
    $this->uuidService = $uuid_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.block'),
      $container->get('context.repository'),
      $container->get('uuid')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state): array {

    // Ajax add more button.
    $form['tabs'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'block-tabs',
      ],
    ];

    // Get tabs using uuids.
    $tabs = $form_state->get('tabs');
    if (empty($tabs) && !$form_state->isRebuilding()) {
      $form_state->set('tabs', $this->configuration['tabs'] ?? []);
      $tabs = $form_state->get('tabs');
    }

    // Get blocks definition.
    $definitions = $this->blockManager->getDefinitionsForContexts($this->contextRepository->getAvailableContexts());
    $options = [];
    foreach ($definitions as $id => $definition) {
      $options[$id] = $definition['admin_label'];
    }
    foreach ($tabs as $uuid => $tab) {
      $form['tabs'][$uuid] = [
        '#type' => 'details',
        '#title' => $this->t('Tab'),
      ];

      if (!empty($tab['title'])) {
        $form['tabs'][$uuid]['#title'] = $this->t('Tab: @title', ['@title' => $tab['title']]);
      }

      $form['tabs'][$uuid]['block_reference'] = [
        '#type' => 'select',
        '#title' => $this->t('Block'),
        '#options' => $options,
        '#default_value' => $tab['block_reference'] ?? '',
      ];

      // REmove button.
      $form['tabs'][$uuid]['remove'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove'),
        '#ajax' => [
          'callback' => [$this, 'rebuildTabs'],
          'wrapper' => 'block-tabs',
          'event' => 'click',
        ],
        '#submit' => [static::class . '::removeCallback'],
        '#name' => $uuid,
      ];
    }

    $form['add'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add'),
      '#ajax' => [
        'callback' => [$this, 'rebuildTabs'],
        'wrapper' => 'block-tabs',
        'event' => 'click',
      ],
      '#submit' => [[$this, 'addMoreCallback']],
    ];

    return $form;
  }

  /**
   * Ajax callback for adding more tabs.
   */
  public function rebuildTabs(array &$form, FormStateInterface $form_state): array {
    $form_state->setRebuild(TRUE);
    return $form["settings"]["tabs"];
  }

  /**
   * Form submission handler for the 'add' button.
   */
  public function addMoreCallback(array &$form, FormStateInterface $form_state): void {
    $tabs = $form_state->get('tabs');
    $uuid = $this->uuidService->generate();
    $tabs[$uuid] = ['block_reference' => ''];
    $form_state->set('tabs', $tabs);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Form submission handler for the 'remove' button.
   */
  public static function removeCallback(array &$form, FormStateInterface $form_state): void {
    $tabs = $form_state->get('tabs');
    $uuid = $form_state->getTriggeringElement()['#name'];
    unset($tabs[$uuid]);
    $form_state->set('tabs', $tabs);
    $form_state->setRebuild(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state): void {
    // Save the configuration.
    $this->configuration['tabs'] = $form_state->getValue('tabs');
  }

  /**
   * Build a block.
   */
  protected function buildBlock(string $blockReference): array {
    // @todo Configurations are currently not passed to the block.
    $block = $this->blockManager->createInstance($blockReference);
    $block_title = $block->label();
    if ($block instanceof ViewsBlock) {
      $block_title = $block->getViewExecutable()->getTitle();
    }

    return [
      'title' => $block_title,
      'content' => $block->build(),
    ];
  }

  /**
   * Build the tabs.
   */
  protected function buildTabs(): array {
    $tabs = $this->configuration['tabs'] ?? [];
    $build = [];
    foreach ($tabs as $tab) {
      $build[] = $this->buildBlock($tab['block_reference']);
    }
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    return [
      '#theme' => 'blocks_tabs',
      '#tabs' => $this->buildTabs(),
    ];
  }

}
