<?php

namespace Drupal\seeds_widgets\Plugin\Field\FieldFormatter;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceFormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\seeds_widgets\SeedsWidgetsManager;
use Drupal\seeds_widgets\SeedsWidgetTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\seeds_widgets\Service\SwiperOptionsNormalizer;

/**
 * Plugin implementation of the 'carousel' formatter.
 *
 * @FieldFormatter(
 *   id = "seeds_widgets_carousel",
 *   label = @Translation("Seeds Widgets Carousel"),
 *   field_types = {
 *     "entity_reference_revisions"
 *   }
 * )
 */
class CarouselFormatter extends EntityReferenceFormatterBase implements ContainerFactoryPluginInterface, SeedsWidgetTypeInterface {

  /**
   * The Seeds Widgets Manager.
   *
   * @var \Drupal\seeds_widgets\SeedsWidgetsManager
   */
  protected $seedsWidgetsManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The Swiper Options Normalizer service.
   *
   * @var \Drupal\seeds_widgets\Service\SwiperOptionsNormalizer
   */
  protected $swiperOptionsNormalizer;

  /**
   * Constructs a CarouselFormatter object.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\seeds_widgets\SeedsWidgetsManager $seeds_widgets_manager
   *   The Seeds Widgets Manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\seeds_widgets\Service\SwiperOptionsNormalizer $swiper_options_normalizer
   *   The Swiper Options Normalizer service.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, SeedsWidgetsManager $seeds_widgets_manager, EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory, SwiperOptionsNormalizer $swiper_options_normalizer) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->seedsWidgetsManager = $seeds_widgets_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
    $this->swiperOptionsNormalizer = $swiper_options_normalizer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('seeds_widgets.manager'),
      $container->get('entity_type.manager'),
      $container->get('config.factory'),
      $container->get('seeds_widgets.swiper_options_normalizer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'swiper_template' => 'default',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);
    $settings = $this->getSettings();

    // Get available swiper templates.
    $templates = $this->entityTypeManager
      ->getStorage('swiper_formatter')
      ->loadMultiple();

    $template_options = [];
    foreach ($templates as $template) {
      $template_options[$template->id()] = $template->label();
    }

    $form['swiper_template'] = [
      '#type' => 'select',
      '#title' => $this->t('Swiper Template'),
      '#options' => $template_options,
      '#default_value' => $settings['swiper_template'] ?? 'default',
      '#description' => $this->t('Select a swiper template to use.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();
    $settings = $this->getSettings();

    // Get the template name.
    $template_id = $settings['swiper_template'] ?? 'default';
    $template = $this->entityTypeManager
      ->getStorage('swiper_formatter')
      ->load($template_id);

    if ($template) {
      $summary[] = $this->t('Using swiper template: @template', [
        '@template' => $template->label(),
      ]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    if ($items->isEmpty()) {
      return $elements;
    }

    $settings = $this->getSettings();
    $entities = $this->getEntitiesToView($items, $langcode);
    $processed_items = [];

    foreach ($entities as $delta => $entity) {
      if (!$entity) {
        continue;
      }

      $view_builder = $this->entityTypeManager->getViewBuilder($entity->getEntityTypeId());

      // Check if the paragraph has a custom view mode field and use that.
      $view_mode = $this->viewMode;
      if ($entity instanceof ContentEntityInterface &&
          $entity->hasField('field_seeds_widget_view_mode') &&
          !$entity->get('field_seeds_widget_view_mode')->isEmpty()) {
        $custom_view_mode = $entity->get('field_seeds_widget_view_mode')->target_id;
        if ($custom_view_mode) {
          $view_mode = str_replace("paragraph.", "", $custom_view_mode);
        }
      }

      $processed_items[$delta] = [
        'content' => $view_builder->view($entity, $view_mode, $entity->language()->getId()),
        '#paragraph' => $entity,
      ];
    }

    // Get swiper template configuration.
    $template_id = $settings['swiper_template'] ?? 'default';
    /** @var \Drupal\swiper_formatter\Entity\SwiperFormatterInterface $template */
    $template = $this->entityTypeManager
      ->getStorage('swiper_formatter')
      ->load($template_id);

    if ($template) {
      $swiper_options = $template->get('swiper_options') ?? [];
    }
    else {
      $swiper_options = $this->configFactory->get('swiper_formatter.swiper_formatter.default')
        ->get('swiper_options');
    }

    // Normalize swiper breakpoints.
    $swiper_options = $this->swiperOptionsNormalizer->normalizeSwiperBreakpoints($swiper_options);

    // Build the base render array with swiper theme.
    $elements[0] = [
      '#theme' => 'seeds_widgets_paragraph_formatter__carousel',
      '#items' => $processed_items,
      '#widget_type' => 'carousel',
      '#settings' => [
        'swiper' => $swiper_options,
      ],
      '#attached' => [
        'library' => [
          'swiper_formatter/swiper',
          'seeds_widgets/swiper_widgets',
        ],
      ],
    ];

    return $elements;
  }

}
