<?php

namespace Drupal\seeds_widgets;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Image\ImageFactory;
use Drupal\Core\Render\Renderer;
use Drupal\Core\Routing\CurrentRouteMatch;

/**
 * Seeds Widgets Manager service.
 */
class SeedsWidgetsManager {

  /**
   * Block types provided by Seeds.
   */
  protected const PARAGRAPH_BLOCKS = [
    'seeds_widget',
    'seeds_table',
    'seeds_html',
    'seeds_files',
    'seeds_gallery',
  ];

  /**
   * Accepted entity types supported by Seeds.
   */
  protected const ACCEPTED_ENTITY_TYPES = [
    'node',
    'media',
    'taxonomy_term',
    'user',
  ];

  /**
   * Renderer.
   *
   * @var \Drupal\Core\Render\Renderer
   */
  protected $renderer;

  /**
   * Image Factory.
   *
   * @var \Drupal\Core\Image\ImageFactory
   */
  protected $imageFactory;

  /**
   * Current Route Match.
   *
   * @var \Drupal\Core\Routing\CurrentRouteMatch
   */
  protected $routeMatch;

  /**
   * Config Factory definition.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Entity Display Repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected $entityDisplayRepository;

  /**
   * Construct new SeedsWidgetsManager object.
   *
   * @param \Drupal\Core\Render\Renderer $renderer
   *   The renderer service.
   * @param \Drupal\Core\Image\ImageFactory $image_factory
   *   The image factory service.
   * @param \Drupal\Core\Routing\CurrentRouteMatch $route_match
   *   The current route match service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entity_display_repository
   *   The entity display repository service.
   */
  public function __construct(
    Renderer $renderer,
    ImageFactory $image_factory,
    CurrentRouteMatch $route_match,
    ConfigFactoryInterface $config_factory,
    EntityDisplayRepositoryInterface $entity_display_repository,
  ) {
    $this->imageFactory = $image_factory;
    $this->renderer = $renderer;
    $this->routeMatch = $route_match;
    $this->configFactory = $config_factory;
    $this->entityDisplayRepository = $entity_display_repository;
  }

  /**
   * Checks if the passed entity is a Block and is of certain type.
   *
   * @param mixed $entity
   *   The entity to be checked.
   * @param string|null $type
   *   [Optional] The type of the block.
   *
   * @return bool
   *   TRUE if the entity is a Seeds block, FALSE otherwise.
   */
  public static function isSeedsBlock($entity, $type = NULL) {
    if ($type) {
      return $entity && $entity->getEntityTypeId() === 'block_content' && $entity->bundle() === $type;
    }

    return $entity && $entity->getEntityTypeId() === 'block_content' && in_array($entity->bundle(), self::PARAGRAPH_BLOCKS, TRUE);
  }

  /**
   * Checks if the passed entity type id is supported by Seeds.
   *
   * @param string $entity_type_id
   *   The id of the entity type to be checked.
   *
   * @return bool
   *   TRUE if the entity type is accepted, FALSE otherwise.
   */
  public function isAcceptedEntity($entity_type_id) {
    return in_array($entity_type_id, self::ACCEPTED_ENTITY_TYPES, TRUE);
  }

  /**
   * Gets the rendered array of an image with a responsive image style applied.
   *
   * @param \Drupal\file\Entity\File|null $file
   *   The image file.
   * @param string|null $responsive_image_id
   *   The responsive image style to be applied.
   *
   * @return array|\Drupal\Component\Render\MarkupInterface|mixed|string|void
   *   The rendered image or empty array.
   *
   * @throws \Exception
   */
  public function toResponsiveImage($file, $responsive_image_id) {
    if (!$file || !$responsive_image_id) {
      return [];
    }

    $file_uri = $file->getFileUri();

    $image = $this->imageFactory->get($file->getFileUri());

    if ($image->isValid()) {
      $width = $image->getWidth();
      $height = $image->getHeight();
    }
    else {
      $width = $height = NULL;
    }

    $image_build = [
      '#theme' => 'responsive_image',
      '#width' => $width,
      '#height' => $height,
      '#responsive_image_style_id' => $responsive_image_id,
      '#uri' => $file_uri,
    ];

    $this->renderer->addCacheableDependency($image_build, $file);
    return $this->renderer->render($image_build);
  }

  /**
   * Build allowed view modes option list to be used when creating a paragraph.
   *
   * @param string $paragraph_bundle
   *   The bundle of the paragraph that is being created.
   *
   * @return array
   *   An array of allowed view mode options.
   */
  public function getAllowedViewModesOptions($paragraph_bundle) {
    $paragraph_view_modes = $this->entityDisplayRepository->getViewModeOptionsByBundle('paragraph', $paragraph_bundle);
    $allowed_view_modes = $this->configFactory->get('seeds_widgets.paragraph_view_modes')->get('allowed_view_modes');
    $options = [];

    foreach ($allowed_view_modes as $key => $value) {
      if ($value !== 0 && array_key_exists($key, $paragraph_view_modes)) {
        if ($key === 'default') {
          $options["_none"] = $paragraph_view_modes[$key];
        }
        else {
          $options["paragraph.$key"] = $paragraph_view_modes[$key];
        }
      }
    }

    return $options;
  }

}
