<?php

namespace Drupal\seeds_widgets\Service;

use Drupal\swiper_formatter\Service\Swiper;

/**
 * Service to normalize Swiper breakpoints.
 */
class SwiperOptionsNormalizer {

  /**
   * The Swiper service.
   *
   * @var \Drupal\swiper_formatter\Service\Swiper
   */
  protected $swiperService;

  /**
   * Constructs a SwiperOptionsNormalizer object.
   *
   * @param \Drupal\swiper_formatter\Service\Swiper $swiper_service
   *   The Swiper service.
   */
  public function __construct(Swiper $swiper_service) {
    $this->swiperService = $swiper_service;
  }

  /**
   * Normalizes Swiper breakpoints.
   *
   * @param array $swiper_options
   *   The Swiper options array.
   *
   * @return array
   *   The normalized Swiper options.
   */
  public function normalizeSwiperBreakpoints(array $swiper_options) {
    if (empty($swiper_options['breakpoints']) || !is_array($swiper_options['breakpoints'])) {
      return $swiper_options;
    }
    $normalized_breakpoints = [];
    // Only allow valid breakpoint override options.
    $allowed_keys = [
      'slidesPerView',
      'spaceBetween',
      'slidesPerGroup',
      'grid',
    ];

    foreach ($swiper_options['breakpoints'] as $breakpoint_config) {
      if (empty($breakpoint_config['breakpoint']) || empty($breakpoint_config['swiper_template'])) {
        continue;
      }

      $breakpoint = (int) $breakpoint_config['breakpoint'];
      /** @var \Drupal\swiper_formatter\SwiperFormatterInterface $bp_template */
      $bp_template = $this->swiperService->getSwiper($breakpoint_config['swiper_template']);

      if (!$bp_template) {
        continue;
      }

      $bp_options = $bp_template->getSwiperOptions() ?? [];
      // Filter ONLY valid Swiper breakpoint overrides.
      $bp_options = array_intersect_key(
        $bp_options,
        array_flip($allowed_keys)
      );

      if (!empty($bp_options)) {
        $normalized_breakpoints[$breakpoint] = $bp_options;
      }
    }

    if (!empty($normalized_breakpoints)) {
      ksort($normalized_breakpoints);
      $swiper_options['breakpoints'] = $normalized_breakpoints;
    }

    return $swiper_options;
  }

}
