<?php

namespace Drupal\select_a11y_ng\Element;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element\Select;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides a select_a11y_ng form element.
 *
 * @FormElement("select_a11y_ng")
 */
class SelectA11yNG extends Select {

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    $info = parent::getInfo();
    $class = get_class($this);

    // Apply default form element properties.
    $info['#cardinality'] = 0;
    $info['#pre_render'][] = [$class, 'preRenderOverwrites'];
    $info['#select_a11y_ng'] = [];

    return $info;
  }

  /**
   * {@inheritdoc}
   */
  public static function valueCallback(&$element, $input, FormStateInterface $form_state) {
    // Potentially the #value is set directly, so it contains the 'target_id'
    // array structure instead of a string.
    if ($input !== FALSE && is_array($input)) {
      $input = array_map(function ($item) {
        return $item['target_id'] ?? $item;
      }, $input);
      return array_combine($input, $input);
    }

    return parent::valueCallback($element, $input, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public static function processSelect(&$element, FormStateInterface $form_state, &$complete_form) {
    $element = parent::processSelect($element, $form_state, $complete_form);

    // Set the type from select_a11y_ng to select to get proper form validation.
    $element['#type'] = 'select';

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function preRenderSelect($element) {
    $element = parent::preRenderSelect($element);
    $multiple = $element['#multiple'];

    if ($multiple) {
      $element['#attributes']['multiple'] = 'multiple';
      $element['#attributes']['name'] = $element['#name'] . '[]';
    }

    $current_language = \Drupal::languageManager()->getCurrentLanguage();

    // Placeholder should be taken from #placeholder property if it set.
    $placeholder_text = new TranslatableMarkup('Search in list');
    $placeholder = $placeholder_text;
    if (!empty($element['#placeholder'])) {
      $placeholder = $element['#placeholder'];
    }

    // Defining the select_a11y_ng configuration.
    $settings = [
      'multiple' => $multiple,
      'placeholder' => $placeholder,
      'dir' => $current_language->getDirection(),
      'language' => $current_language->getId(),
    ];

    $element['#attributes']['class'][] = 'select-a11y-ng-widget';
    $element['#attributes']['data-select-a11y-ng-config'] = $settings;

    // Adding the select_a11y_ng library.
    $element['#attached']['library'][] = 'select_a11y_ng/select_a11y_ng.widget';

    return $element;
  }

  /**
   * Allows to modify the select_a11y_ng settings.
   */
  public static function preRenderOverwrites($element) {
    // Allow overwriting the default settings and set additional settings.
    foreach ($element['#select_a11y_ng'] as $key => $value) {
      $element['#attributes']['data-select-a11y-ng-config'][$key] = $value;
    }
    $element['#attributes']['data-select-a11y-ng-config'] = Json::encode($element['#attributes']['data-select-a11y-ng-config']);

    return $element;
  }

}
