<?php

namespace Drupal\select_a11y_ng_webform\Plugin\WebformElement;

use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Plugin\WebformElement\OptionsBase;
use Drupal\webform\Plugin\WebformElement\Select;
use Drupal\webform\WebformSubmissionInterface;

/**
 * Provides a 'select_a11y_ng' element.
 *
 * @WebformElement(
 *   id = "select_a11y_ng",
 *   label = @Translation("Select A11y NG"),
 *   description = @Translation("Provides an accessible multiple select widget."),
 *   category = @Translation("Options elements"),
 * )
 */
class WebformSelectA11yNG extends Select {

  /**
   * {@inheritdoc}
   */
  protected function defineDefaultProperties() {
    $properties = [
      // Select A11y NG specific settings.
      'search' => TRUE,
    ] + parent::defineDefaultProperties();

    // Remove select2, chosen, and choices as they conflict with select_a11y_ng.
    unset($properties['select2']);
    unset($properties['chosen']);
    unset($properties['choices']);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function prepare(array &$element, ?WebformSubmissionInterface $webform_submission = NULL) {
    // Set the element type to select_a11y_ng.
    $element['#type'] = 'select_a11y_ng';

    // Configure Select A11y NG specific settings.
    $element['#select_a11y_ng'] = [];

    // Set placeholder if provided.
    if (!empty($element['#placeholder'])) {
      $element['#select_a11y_ng']['placeholder'] = $element['#placeholder'];
    }

    // Set search option.
    if (isset($element['#search'])) {
      $element['#select_a11y_ng']['search'] = (bool) $element['#search'];
    }

    // Set cardinality for multiple values.
    if (!empty($element['#multiple'])) {
      $element['#cardinality'] = is_numeric($element['#multiple']) ? (int) $element['#multiple'] : 0;
    }

    // Call parent prepare without select2/chosen/choices processing.
    // We skip the parent's prepare to avoid conflicts with
    // select2/chosen/choices.
    OptionsBase::prepare($element, $webform_submission);
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    // Remove select2, chosen, and choices options.
    unset($form['options']['select2']);
    unset($form['options']['chosen']);
    unset($form['options']['choices']);
    unset($form['options']['select_message']);

    // Add Select A11y NG specific options.
    $form['select_a11y_ng'] = [
      '#type' => 'details',
      '#title' => $this->t('Select A11y NG settings'),
      '#open' => TRUE,
    ];

    $form['select_a11y_ng']['search'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow search'),
      '#description' => $this->t('Adds a search input field to enable filtering of values.'),
      '#return_value' => TRUE,
      '#default_value' => TRUE,
    ];

    // Make placeholder always visible for Select A11y NG.
    if (isset($form['form']['placeholder'])) {
      $form['form']['placeholder']['#access'] = TRUE;
      $form['form']['placeholder']['#states'] = [];
    }

    return $form;
  }

}
