<?php

namespace Drupal\select_a11y_ng\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldWidget\OptionsSelectWidget;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'select_a11y_ng' widget.
 *
 * @FieldWidget(
 *   id = "select_a11y_ng",
 *   label = @Translation("Select a11y NG"),
 *   field_types = {
 *     "list_integer",
 *     "list_float",
 *     "list_string",
 *     "entity_reference"
 *   },
 *   multiple_values = TRUE
 * )
 */
class SelectA11yNGWidget extends OptionsSelectWidget {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'placeholder' => (string) t('- None -'),
      'search' => TRUE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element['placeholder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Placeholder text'),
      '#required' => TRUE,
      '#description' => $this->t('Text to be shown in the Select A11y field until a value is selected.'),
      '#default_value' => $this->getSetting('placeholder'),
    ];

    $element['search'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow search'),
      '#description' => $this->t('Adds a search input field to enable filtering of values.'),
      '#default_value' => $this->getSetting('search'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = $this->t('Placeholder: @placeholder', ['@placeholder' => $this->getSetting('placeholder')]);
    $summary[] = $this->t('Search enabled: @search', ['@search' => $this->getSetting('search') ? $this->t('Yes') : $this->t('No')]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $element['#type'] = 'select_a11y_ng';
    $element['#cardinality'] = $this->fieldDefinition->getFieldStorageDefinition()->getCardinality();
    $element['#select_a11y_ng'] = [
      'placeholder' => $this->getSetting('placeholder'),
      'search' => (bool) $this->getSetting('search'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEmptyLabel() {}

  /**
   * {@inheritdoc}
   */
  public static function validateElement(array $element, FormStateInterface $form_state) {
    if ($element['#required'] && $element['#value'] == '') {
      $form_state->setError($element, t('@name field is required.', ['@name' => $element['#title']]));
    }

    // Massage submitted form values.
    // Drupal\Core\Field\WidgetBase::submit() expects values as
    // an array of values keyed by delta first, then by column, while our
    // widgets return the opposite.
    if (is_array($element['#value'])) {
      $values = array_values($element['#value']);
    }
    else {
      $values = [$element['#value']];
    }

    // Filter out the '' option. Use a strict comparison, because
    // 0 == 'any string'.
    $index = array_search('', $values, TRUE);
    if ($index !== FALSE) {
      unset($values[$index]);
    }

    $items = static::prepareFieldValues($values, $element);
    $form_state->setValueForElement($element, $items);
  }

  /**
   * Set's the values to the correct column key.
   *
   * @param array $values
   *   The input values.
   * @param array $element
   *   The render element.
   *
   * @return array
   *   Values with the correct keys.
   */
  protected static function prepareFieldValues(array $values, array $element) {
    // Transpose selections from field => delta to delta => field.
    $items = [];
    foreach ($values as $value) {
      $items[] = [$element['#key_column'] => $value];
    }
    return $items;
  }

}
