<?php

namespace Drupal\selective_contextual_edit\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\selective_contextual_edit\Ajax\ReloadPageCommand;
use Symfony\Component\DependencyInjection\ContainerInterface;

class SingleFieldEntityForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new SingleFieldEntityForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'single_field_entity_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $entity = NULL, $field_name = NULL, $form_mode = NULL) {
    /** @var ContentEntityInterface $entity */

    $form_display = \Drupal::service('entity_display.repository')
        ->getFormDisplay($entity->getEntityTypeId(), $entity->bundle(), $form_mode);

    $wrapper_id = "sfef-{$entity->getEntityTypeId()}-{$entity->id()}-{$field_name}";
    $form_state->set('wrapper_id', $wrapper_id);

    $form += ['#parents' => []];
    $form['#prefix'] = '<div id="' . $wrapper_id . '">';
    $form['#suffix'] = '</div>';

    $form['message_container'] = [
      '#type' => 'container',
      '#attributes' => ['id' => $wrapper_id . '-ajax-message-container'], // Unique ID for targeting
    ];

    if ($widget = $form_display->getRenderer($field_name)) {
      $items = $entity->get($field_name);
      $items->filterEmptyItems();
      $form[$field_name] = $widget->form($items, $form, $form_state);
      $form[$field_name]['#access'] = $items->access('edit');
    }

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#ajax' => [
        'callback' => '::ajaxSubmitCallback',
        'event' => 'click',
        'progress' => [
          'type' => 'throbber',
        ],
      ],
    ];

    $form_state->set('entity_type_id', $entity->getEntityTypeId());
    $form_state->set('entity', $entity);
    $form_state->set('field_name', $field_name);
    $form_state->set('form_mode', $form_mode);
  
    $field_label =  $entity->get($field_name)->getFieldDefinition()->getLabel();
    $form_state->set('field_label', $field_label);

    $current_url = \Drupal::request()->server->get('HTTP_REFERER');
    $form_state->set('current_url', $current_url);

    $form['#attached']['library'][] = 'selective_contextual_edit/contextual_edit';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $entity_type_id = $form_state->get('entity_type_id');
    $entity = $form_state->get('entity');
    $field_name = $form_state->get('field_name');
    $form_mode = $form_state->get('form_mode');

    /** @var EntityFormDisplay $form_display */
    $form_display = \Drupal::service('entity_display.repository')
        ->getFormDisplay($entity_type_id, $entity->bundle(), $form_mode);

    // extractFormValues
    /** @var \Drupal\Core\Field\WidgetBaseInterface */
    $widget = $form_display->getRenderer($field_name);
    $widget->extractFormValues($entity->get($field_name), $form, $form_state);

    $field = $entity->get($field_name);
    if (method_exists($field, 'referencedEntities')) {
      $referencedEntities = $field->referencedEntities();
      foreach ($referencedEntities as $referencedEntity) {
        $referencedEntity->save();
      }
    }

    // Save the entity.
    $entity->save();

    $this->messenger()->addStatus($this->t('The field "%field_label" of "%entity" has been updated.', [
      '%field_label' => $form_state->get('field_label'),
      '%entity' => $entity->label(),
    ]));
  }

  public function ajaxSubmitCallback(array &$form, FormStateInterface $form_state) {
      $response = new AjaxResponse();
      if ($form_state->hasAnyErrors()) {
        $wrapper_id = $form_state->get('wrapper_id');
        $response->addCommand(new ReplaceCommand("#{$wrapper_id}", $form));
        $messenger = \Drupal::messenger();

        // Render the status messages.
        $messages = [
          '#theme' => 'status_messages',
          '#message_list' => $messenger->all(),
          '#status_headings' => [
            'status' => $this->t('Status message'),
            'error' => $this->t('Error message'),
            'warning' => $this->t('Warning message'),
          ],
        ];

        // Replace the content of the message container with the rendered messages.
        $response->addCommand(new ReplaceCommand("#{$wrapper_id}-ajax-message-container", \Drupal::service('renderer')->render($messages)));
        $messenger->deleteAll();
      }
      else {
        $response->addCommand(new CloseModalDialogCommand());
        $response->addCommand(new ReloadPageCommand());
      }
      return $response;
    }

}