<?php

namespace Drupal\seo_analyzer;

use InvalidArgumentException;
use ReflectionException;
use Drupal\seo_analyzer\Page;
use Drupal\seo_analyzer\HttpClient\Client;
use Drupal\seo_analyzer\HttpClient\ClientInterface;
use Drupal\seo_analyzer\HttpClient\Exception\HttpException;
use Drupal\seo_analyzer\Metric\MetricFactory;
use Drupal\seo_analyzer\Metric\MetricInterface;
use Drupal\Core\Utility\Error;

class Analyzer {
  /**
   * @var Page Web page to analyze
   */
  public $page;

  /**
   * @var string Default langcode to use for translations
   */
  public $langcode = 'en';

  /**
   * @var array Metrics array
   */
  public $metrics = [];

  /**
   * @var ClientInterface
   */
  public $client;

  /**
   * Set some background classes based on the negative impact value.
   */
  protected $backgroundClasses = [
    0 => 'bg-success',
    1 => 'bg-warning',
    2 => 'bg-warning',
    3 => 'bg-warning',
    4 => 'bg-warning',
    5 => 'bg-warning',
    6 => 'bg-danger',
    7 => 'bg-danger',
    8 => 'bg-danger',
    9 => 'bg-danger',
    10 => 'bg-danger',
  ];

  /**
   * @param Page|null $page Page to analyze
   * @param ClientInterface|null $client
   */
  public function __construct(Page $page = NULL, ClientInterface $client = NULL) {
    $this->client = $client;
    if (empty($client)) {
      $this->client = new Client();
    }

    if (!empty($page)) {
      $this->page = $page;
    }
  }

  /**
   * Generate a SEO keyword analyze page render array.
   * 
   * @param string $url_string
   *   Full url string: https://mysite.com/nl/demo
   * @param string $url_without_scheme
   *   Url without scheme: mysite.com/nl/demo
   * @param string $langcode
   *   The langcode of the page content of the given url.
   *
   * @return array
   *   A content render array.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   * @throws ReflectionException
   */
  public function generateAnalyzerPage($url_string, $url_without_scheme, $langcode) {
    // Fallback keyword.
    $keyword = 'keyword';
    // Get keyword from query parameter.
    if (isset($_GET['keyword']) && !empty($_GET['keyword'])) {
      $keyword = $_GET['keyword'];
    }

    try {
      $results = $this->analyzeUrl($url_string, $keyword, $langcode);
    }
    catch (HttpException $e) {
      // Log the exception to watchdog.
      Error::logException(\Drupal::logger('seo_analyzer'), $e, 'Error loading page');
      // Show an error page.
      return [
        '#theme' => 'page_seo_analyzer_error',
        '#message' => $e->getMessage(),
        '#attached' => ['library' => ['seo_analyzer/styling']],
      ];
    }
    catch (ReflectionException $e) {
      // Log the exception to watchdog.
      Error::logException(\Drupal::logger('seo_analyzer'), $e, 'Error loading metric file');
      // Show an error page.
      return [
        '#theme' => 'page_seo_analyzer_error',
        '#message' => $e->getMessage(),
        '#attached' => ['library' => ['seo_analyzer/styling']],
      ];
    }

    // Create and return a render array.
    return [
      '#theme' => 'page_seo_analyzer',
      '#search_form' => \Drupal::formBuilder()->getForm('Drupal\seo_analyzer\Form\KeywordForm'),
      '#keyword_analytics' => $this->createKeywordAnalyticsForm($results, $keyword),
      '#content_analytics' => $this->createContentAnalyticsForm($results, $url_string, $url_without_scheme),
      '#general_analytics' => $this->createGeneralAnalyticsForm($results, $url_string, $url_without_scheme),
      '#attached' => ['library' => ['seo_analyzer/styling']],
    ];
  }

  /**
   * Analyzes page at specified url.
   *
   * @param string $url Url to analyze
   * @param string|null $keyword
   * @param string|null $langcode
   * @return array
   * @throws ReflectionException
   */
  public function analyzeUrl(string $url, string $keyword = NULL, string $langcode = NULL): array {
    if (!empty($langcode)) {
      $this->langcode = $langcode;
    }
    $this->page = new Page($url, $langcode, $this->client);
    if (!empty($keyword)) {
      $this->page->keyword = $keyword;
    }
    return $this->analyze();
  }

  /**
   * Analyzes html document from file.
   *
   * @param string $filename
   * @param string|null $langcode
   * @return array
   * @throws ReflectionException
   */
  public function analyzeFile(string $filename, string $langcode = NULL): array {
    $this->page = new Page(NULL, $langcode, $this->client);
    $this->page->content = file_get_contents($filename);
    return $this->analyze();
  }

  /**
   * Analyzes html document from string.
   *
   * @param string $htmlString
   * @param string|null $langcode
   * @return array
   * @throws ReflectionException
   */
  public function analyzeHtml(string $htmlString, string $langcode = NULL): array {
    $this->page = new Page(NULL, $langcode, $this->client);
    $this->page->content = $htmlString;
    return $this->analyze();
  }

  /**
   * Starts analysis of a Page.
   *
   * @return array
   * @throws ReflectionException
   */
  public function analyze() {

    if (empty($this->page)) {
      throw new InvalidArgumentException('No Page to analyze');
    }
    if (empty($this->metrics)) {
      $this->metrics = $this->getMetrics();
    }
    $results = [];
    foreach ($this->metrics as $metric) {
      if ($analysisResult = $metric->analyze()) {
        $results[$metric->name] = $this->formatResults($metric, $analysisResult);
      }
    }
    return $results;
  }

  /**
   * Returns available metrics list for a Page
   *
   * @return array
   * @throws ReflectionException
   */
  public function getMetrics(): array {
    return array_merge($this->page->getMetrics(), $this->getFilesMetrics());
  }

  /**
   * Returns file related metrics.
   *
   * @return array
   * @throws ReflectionException
   */
  public function getFilesMetrics(): array {
    return [
      'robots' => MetricFactory::get('file.robots', $this->getFileContent(
        $this->page->getFactor('url.parsed.scheme') . '://' . $this->page->getFactor('url.parsed.host'),
        'robots.txt'
      )),
      'sitemap' => MetricFactory::get('file.sitemap', $this->getFileContent(
        $this->page->getFactor('url.parsed.scheme') . '://' . $this->page->getFactor('url.parsed.host'),
        'sitemap.xml'
      ))
    ];
  }

  /**
   * Downloads file from Page's host.
   *
   * @param $url
   * @param $filename
   * @return bool|string
   */
  protected function getFileContent($url, $filename) {
    try {
      $response = $this->client->get($url . '/' . $filename);
      $content = $response->getBody()->getContents();
    }
    catch (HttpException $e) {
      return FALSE;
    }
    return $content;
  }

  /**
   * Formats metric analysis results.
   *
   * @param MetricInterface $metric
   * @param $results
   * @return array
   */
  protected function formatResults(MetricInterface $metric, string $results): array {

    return [
      'analysis' => $results,
      'name' => $metric->name,
      'description' => $metric->description,
      'value' => $metric->value,
      'negative_impact' => $metric->impact,
    ];
  }

  /**
   * Display the seo analytics for a given keyword.
   *
   * @param array $results
   *   Array returned by the analyzeUrl() function.
   * @param string $keyword
   *   The keyword that will be used to check SEO for the content against.
   *
   * @return array
   *   A render array that displays the seo analytics.
   */
  private function createKeywordAnalyticsForm($results, $keyword) {
    $keyword_analytics_form = [];
    $keyword_analytics_form['keyword_analytics'] = [
      '#type' => 'details',
      '#title' => t("Keyword analytics for '@keyword'", ['@keyword' => $keyword]),
      '#open' => TRUE,
    ];
    $keyword_analytics_form['keyword_analytics']['content'] = [
      '#theme' => 'table_keyword_analytics',
      '#bgclasses' => $this->backgroundClasses,
      '#data' => [
        'meta_title' => $results['KeywordTitle'],
        'meta_description' => $results['KeywordDescription'],
        'heading_keywords' => $results['PageKeywordHeaders'],
        'heading_keyword_density' => $results['PageHeadersKeywordDensity'],
        'content_keyword_density' => $results['KeywordDensityKeyword'],
        'path_keyword' => $results['KeywordUrlPath'],
        'domain_keyword' => $results['KeywordURL'],
      ],
    ];

    return $keyword_analytics_form;
  }

  /**
   * Display the content analytics for a node's content.
   *
   * @param array $results
   *   Array returned by the analyzeUrl() function.
   * @param string $url
   *   Full node url string: https://mysite.com/nl/demo
   * @param string $page_url
   *   Node url string without scheme: mysite.com/nl/demo
   *
   * @return array
   *   A render array that displays the seo analytics.
   */
  private function createContentAnalyticsForm($results, $url, $page_url) {
    $content_analytics_form = [];
    $content_analytics_form['content_analytics'] = [
      '#type' => 'details',
      '#title' => t("Content analytics for @url", ['@url' => $url]),
      '#open' => TRUE,
    ];
    $content_analytics_form['content_analytics']['content'] = [
      '#theme' => 'table_content_analytics',
      '#bgclasses' => $this->backgroundClasses,
      '#page_url' => $page_url,
      '#data' => [
        'meta_tag_length' => $results['PageMeta'],
        'heading_structure' => $results['PageHeaders'],
        'page_content_ratio' => $results['PageContentRatio'],
        'image_alt_texts' => $results['PageAlts'],
        'page_url_size' => $results['PageUrlLength'],
      ],
    ];

    return $content_analytics_form;
  }

  /**
   * Display the general seo analytics for a node.
   *
   * @param array $results
   *   Array returned by the analyzeUrl() function.
   * @param string $url
   *   Full node url string: https://mysite.com/nl/demo
   * @param string $page_url
   *   Node url string without scheme: mysite.com/nl/demo
   *
   * @return array
   *   A render array that displays the seo analytics.
   */
  private function createGeneralAnalyticsForm($results, $url, $page_url) {
    $general_analytics_form = [];
    $general_analytics_form['general_analytics'] = [
      '#type' => 'details',
      '#title' => t("General analytics for @url", ['@url' => $url]),
      '#open' => TRUE,
    ];
    $general_analytics_form['general_analytics']['content'] = [
      '#theme' => 'table_general_analytics',
      '#bgclasses' => $this->backgroundClasses,
      '#page_url' => $page_url,
      '#data' => [
        'is_encrypted' => $results['PageSSL'],
        'has_redirect' => $results['PageRedirect'],
        'content_size' => $results['PageContentSize'],
        'page_load_time' => $results['PageLoadTime'],
        'robots_txt' => $results['FileRobots'],
        'sitemap' => $results['FileSitemap'],
      ],
    ];

    return $general_analytics_form;
  }
}
