<?php

namespace Drupal\seo_analyzer\Controller;

use Drupal\canvas\Entity\Page;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Drupal\seo_analyzer\Analyzer;

/**
 * Class SeoAnalyzerController.
 *
 * @package Drupal\seo_analyzer\Controller
 */
class SeoAnalyzerController extends ControllerBase {

  /**
   * Instance of the Analyzer class.
   *
   * @var \Drupal\seo_analyzer\Analyzer
   */
  protected $analyzer;

  /**
   * Constructs a SeoAnalyzerController object.
   */
  public function __construct() {
    $this->analyzer = new Analyzer();
  }

  /**
   * Generate the SEO keyword analyze page for a node.
   * 
   * @param \Drupal\node\NodeInterface $node
   *   The Node entity.
   *
   * @return array
   *   A content render array.
   */
  public function generateAnalyzerPageFromNode(NodeInterface $node) {
    // Get current hostname.
    $host = \Drupal::request()->getHost();
    // Get the full node url string: https://mysite.com/nl/demo
    $options = ['absolute' => TRUE];
    $url = Url::fromRoute('entity.node.canonical', ['node' => $node->id()], $options);
    $url_string = $url->toString();
    // Get the node url without scheme: mysite.com/nl/demo
    $url_without_scheme = $host . Url::fromRoute('entity.node.canonical', ['node' => $node->id()], [])->toString();
    // Get node langcode.
    $langcode = $node->get('langcode')->value;

    $result = $this->analyzer->generateAnalyzerPage($url_string, $url_without_scheme, $langcode);
    return $result;
  }

  /**
   * Generate the SEO keyword analyze page for a canvas page.
   * 
   * @param \Drupal\canvas\Entity\Page $canvas_page
   *   The Node entity.
   *
   * @return array
   *   A content render array.
   */
  public function generateAnalyzerPageFromCanvasPage(Page $canvas_page) {
    // Get current hostname.
    $host = \Drupal::request()->getHost();
    // Get the full url string: https://mysite.com/nl/demo
    $options = ['absolute' => TRUE];
    $url = Url::fromRoute('entity.canvas_page.canonical', ['canvas_page' => $canvas_page->id()], $options);
    $url_string = $url->toString();
    // Get the url without scheme: mysite.com/nl/demo
    $url_without_scheme = $host . Url::fromRoute('entity.canvas_page.canonical', ['canvas_page' => $canvas_page->id()], [])->toString();
    // Get langcode.
    $langcode = $canvas_page->get('langcode')->value ?? 'en';
    // Call the Analyzer class to generate the render array.
    $result = $this->analyzer->generateAnalyzerPage($url_string, $url_without_scheme, $langcode);
    return $result;
  }
}
