<?php

declare(strict_types=1);

namespace Drupal\seo_urls\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\seo_urls\Entity\SeoUrlInterface;
use Drupal\seo_urls\SeoUrlManagerInterface;

/**
 * Provides route responses for entity.seo_url.add-form.
 */
class SeoUrlCreateController extends ControllerBase {

  use AutowireTrait;

  public function __construct(
    protected SeoUrlManagerInterface $seoUrlManager,
    protected RouteMatchInterface $routeMatch,
  ) {}

  /**
   * Returns a form to add a new SEO URL.
   *
   * @return array
   *   The SEO URL add form.
   */
  public function addForm(): array {
    /** @var \Drupal\seo_urls\Entity\SeoUrlInterface $entity */
    $entity = $this->entityTypeManager()
      ->getStorage(SeoUrlInterface::ENTITY_TYPE)
      ->create();

    // Set the default canonical Url.
    if ($redirect_path = $this->getRedirectDestination()->get()) {
      $entity->set(SeoUrlInterface::CANONICAL_URL_FIELD, [
        'uri' => 'internal:' . $this->seoUrlManager->clearPathPrefix($redirect_path),
        'title' => '',
        'options' => [],
      ]);
    }

    return $this->entityFormBuilder()->getForm($entity);
  }

  /**
   * Access.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   Whether the current content type is allowed to create SEO Url.
   */
  public function access(): AccessResult {
    $allowed = FALSE;
    $cacheable_dependency = new CacheableMetadata();
    $cacheable_dependency->addCacheTags(['config:seo_urls.settings'])
      ->addCacheContexts(['url.query_args']);

    // Check access for entity.
    $parameters = $this->routeMatch->getParameters()->all();
    $allowed_entity_types = $this->seoUrlManager->getAllowedEntityTypes();
    if ($filtered_parameters = array_intersect_key($parameters, array_fill_keys($allowed_entity_types, TRUE))) {
      foreach ($filtered_parameters as $value) {
        $cacheable_dependency->addCacheableDependency($value);
      }
      $allowed = TRUE;
    }

    // Check access for views.
    if ($this->seoUrlManager->isAllowedForViews()
      && array_key_exists('view_id', $parameters)
      && array_key_exists('display_id', $parameters)
    ) {
      /** @var \Drupal\views\Entity\View $view */
      $view = $this->entityTypeManager()
        ->getStorage('view')
        ->load($parameters['view_id']);
      $cacheable_dependency->addCacheableDependency($view);
      $cacheable_dependency->addCacheTags(['config:core.extension']);
      $allowed = TRUE;
    }

    return AccessResult::allowedIf($allowed || $this->routeMatch->getRouteName() === 'seo_url.create')
      ->addCacheableDependency($cacheable_dependency);
  }

}
