<?php

declare(strict_types=1);

namespace Drupal\seo_urls\EventSubscriber;

use Drupal\Core\Path\CurrentPathStack;
use Drupal\seo_urls\SeoUrlManager;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ControllerEvent;
use Symfony\Component\HttpKernel\Event\TerminateEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Provides a path subscriber that converts SEO urls.
 *
 * @todo Implement caching on the same/similar way as PathAliasSubscriber did.
 * @todo This is a template for now, service wasn't created yet.
 */
class SeoUrlSubscriber implements EventSubscriberInterface {

  public function __construct(
    protected SeoUrlManager $seoUrlManager,
    protected CurrentPathStack $currentPath,
  ) {}

  /**
   * Sets the cache key on the Seo Url manager cache decorator.
   *
   * KernelEvents::CONTROLLER is used in order to be executed after routing.
   *
   * @param \Symfony\Component\HttpKernel\Event\ControllerEvent $event
   *   The Event to process.
   */
  public function onKernelController(ControllerEvent $event): void {
    // Set the cache key on the alias manager cache decorator.
    if ($event->isMainRequest()) {
      // Ignore phpstan check for now because the service is a template for now.
      // @phpstan-ignore-next-line.
      $this->seoUrlManager->setCacheKey(rtrim($this->currentPath->getPath($event->getRequest()), '/'));
    }
  }

  /**
   * Ensures system paths for the request get cached.
   */
  public function onKernelTerminate(TerminateEvent $event): void {
    // Ignore phpstan check for now because the service is a template for now.
    // @phpstan-ignore-next-line.
    $this->seoUrlManager->writeCache();
  }

  /**
   * Registers the methods in this class that should be listeners.
   *
   * @return array
   *   An array of event listener definitions.
   */
  public static function getSubscribedEvents(): array {
    $events[KernelEvents::CONTROLLER][] = ['onKernelController', 200];
    $events[KernelEvents::TERMINATE][] = ['onKernelTerminate', 200];
    return $events;
  }

}
