<?php

declare(strict_types=1);

namespace Drupal\Tests\seo_urls\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\seo_urls\Entity\SeoUrl;

/**
 * Tests the SEO URLs functionality.
 *
 * @group seo_urls
 */
class SeoUrlsTest extends BrowserTestBase {
  /**
   * The default theme to use for testing.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'seo_urls',
    'node',
    'path_alias',
    'block',
    'field',
    'link',
    'text',
    'user',
    'system',
    'path',
    'dynamic_page_cache',
  ];

  /**
   * A test user with administrative privileges.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a content type for testing.
    $this->drupalCreateContentType([
      'type' => 'page',
      'name' => 'Basic page',
    ]);

    // Create an admin user with permissions to manage SEO URLs.
    $permissions = [
      'administer seo_url entities',
      'add seo_url entities',
      'update seo_url entities',
      'view seo_url entities',
      'create page content',
      'edit any page content',
      'create url aliases',
      'administer url aliases',
      'access administration pages',
      'access content',
      'delete seo_url entities',
    ];

    $user = $this->drupalCreateUser($permissions);
    $this->assertNotEmpty($user);
    $this->adminUser = $user;

    // Place the local actions block in the theme.
    $this->placeBlock('local_actions_block');

    // Get the entity type manager service.
    $this->entityTypeManager = $this->container->get('entity_type.manager');
  }

  /**
   * Tests the SEO URL settings form.
   */
  public function testSettingsForm(): void {
    $this->drupalLogin($this->adminUser);

    // Access the settings form.
    $this->drupalGet('admin/structure/seo_url');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('SEO URL');

    // The form might be using a different structure,
    // so we'll just test basic access for now.
    $this->assertSession()->fieldExists('allowed_content_types[node]');
    $this->assertSession()->fieldExists('allowed_content_types[user]');
  }

  /**
   * Tests creating a new SEO URL.
   */
  public function testCreateSeoUrl(): void {
    $this->drupalLogin($this->adminUser);

    // Create a test node.
    $node = $this->drupalCreateNode([
      'type' => 'page',
      'title' => 'Test page',
    ]);

    // Create a new SEO URL.
    $this->drupalGet('admin/seo_url/add');
    $this->assertSession()->statusCodeEquals(200);

    // Check if the form has the expected fields.
    $this->assertSession()->pageTextContains('Create SEO URL');
    $this->assertSession()->fieldExists('canonical_url[0][uri]');
    $this->assertSession()->fieldExists('seo_url[0][uri]');

    $edit = [
      'canonical_url[0][uri]' => '/node/' . $node->id(),
      'seo_url[0][uri]' => '/seo-test-page',
    ];

    // The button might be using a different label or ID.
    $this->submitForm($edit, 'Save');

    // Verify the SEO URL was created.
    $this->assertSession()->pageTextContains('Created /seo-test-page SEO URL.');

    // Verify the SEO URL works.
    $this->drupalGet('/seo-test-page');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Test page');
  }

  /**
   * Tests the SEO URL list builder.
   */
  public function testSeoUrlList(): void {
    $this->drupalLogin($this->adminUser);

    // Create a test node.
    $node = $this->drupalCreateNode([
      'type' => 'page',
      'title' => 'Test page for list',
    ]);

    // Create a test SEO URL.
    $seo_url = SeoUrl::create([
      'canonical_url' => ['uri' => 'internal:/node/' . $node->id()],
      'seo_url' => ['uri' => 'internal:/test-seo-path'],
      'status' => 1,
    ]);
    $seo_url->save();

    // Access the SEO URL list.
    $this->drupalGet('admin/content/seo_url');
    $this->assertSession()->statusCodeEquals(200);

    // The list might show the canonical URL and the SEO URL.
    $this->assertSession()->pageTextContains('/node/' . $node->id());
    $this->assertSession()->pageTextContains('/test-seo-path');
  }

  /**
   * Tests the SEO URL edit form.
   */
  public function testEditSeoUrl(): void {
    $this->drupalLogin($this->adminUser);

    // Create a test node.
    $node = $this->drupalCreateNode([
      'type' => 'page',
      'title' => 'Test page for edit',
    ]);

    // Create a test SEO URL.
    $seo_url = SeoUrl::create([
      'canonical_url' => ['uri' => 'internal:/node/' . $node->id()],
      'seo_url' => ['uri' => 'internal:/test-edit-seo-path'],
      'status' => 1,
    ]);
    $seo_url->save();

    // Edit the SEO URL.
    $this->drupalGet('admin/content/seo_url/' . $seo_url->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);

    // The form might have different field names or structure.
    $this->assertSession()->fieldExists('canonical_url[0][uri]');
    $this->assertSession()->fieldExists('seo_url[0][uri]');

    $edit = [
      'seo_url[0][uri]' => '/updated-seo-path',
    ];

    $this->submitForm($edit, 'Save');

    // Verify the update was successful.
    $this->assertSession()->pageTextContains('Saved /updated-seo-path SEO URL.');
  }

}
