<?php

declare(strict_types=1);

namespace Drupal\Tests\seo_urls\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\seo_urls\SeoUrlManagerInterface;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Symfony\Component\HttpFoundation\Request;

/**
 * Tests the SEO URL manager.
 *
 * @group seo_urls
 */
class SeoUrlManagerTest extends KernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'path_alias',
    'seo_urls',
    'field',
    'text',
    'link',
  ];

  /**
   * The SEO URL manager service.
   *
   * @var \Drupal\seo_urls\SeoUrlManagerInterface
   */
  protected SeoUrlManagerInterface $seoUrlManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('path_alias');
    $this->installEntitySchema('seo_url');
    $this->installConfig(['system', 'seo_urls']);

    $this->seoUrlManager = $this->container->get('seo_urls.manager');
  }

  /**
   * Tests the SEO URL manager service is available.
   */
  public function testServiceAvailability(): void {
    $this->assertInstanceOf(SeoUrlManagerInterface::class, $this->seoUrlManager);
  }

  /**
   * Tests the getPathBySeoUrl method.
   */
  public function testGetPathBySeoUrl(): void {
    // Create a test SEO URL entity.
    $storage = $this->container->get('entity_type.manager')->getStorage('seo_url');
    $seo_url = $storage->create([
      'canonical_url' => ['uri' => 'internal:/test-path'],
      'seo_url' => ['uri' => 'internal:/test-seo-path'],
      'status' => 1,
    ]);
    $seo_url->save();

    // Test getting path by SEO URL.
    $fake_request = Request::create('internal:/test-seo-path');
    $path = $this->seoUrlManager->getCanonicalUrlBySeoUrl('/test-seo-path', $fake_request);
    $this->assertEquals('/test-path', $path);

    // Test with non-existent SEO URL.
    $fake_request = Request::create('internal:/non-existent-path');
    $path = $this->seoUrlManager->getCanonicalUrlBySeoUrl('/non-existent-path', $fake_request);
    $this->assertEquals('/non-existent-path', $path);
  }

  /**
   * Tests the getSeoUrlByPath method.
   */
  public function testGetSeoUrlByPath(): void {
    // Create a test SEO URL entity.
    $storage = $this->container->get('entity_type.manager')->getStorage('seo_url');
    $seo_url = $storage->create([
      'canonical_url' => ['uri' => 'internal:/test-path'],
      'seo_url' => ['uri' => 'internal:/test-seo-path'],
      'status' => 1,
    ]);
    $seo_url->save();

    // Test getting SEO URL by path.
    $fake_request = Request::create('internal:/test-path');
    $seo_path = $this->seoUrlManager->getSeoUrlByCanonicalUrl('/test-path', $fake_request);
    $this->assertEquals('/test-seo-path', $seo_path);

    // Test with non-existent path.
    $fake_request = Request::create('internal:/non-existent-path');
    $seo_path = $this->seoUrlManager->getSeoUrlByCanonicalUrl('/non-existent-path', $fake_request);
    $this->assertEquals('/non-existent-path', $seo_path);
  }

  /**
   * Tests the isSeoUrl method.
   */
  public function testIsSeoUrl(): void {
    // Create a test SEO URL entity.
    $storage = $this->container->get('entity_type.manager')->getStorage('seo_url');
    $seo_url = $storage->create([
      'canonical_url' => ['uri' => 'internal:/test-path'],
      'seo_url' => ['uri' => 'internal:/test-seo-path'],
      'status' => 1,
    ]);
    $seo_url->save();

    // Test with existing SEO URL.
    $this->assertTrue($this->seoUrlManager->isSeoUrl('/test-seo-path'));

    // Test with non-existent SEO URL.
    $this->assertFalse($this->seoUrlManager->isSeoUrl('/non-existent-path'));
  }

  /**
   * Tests the isCanonicalUrl method.
   */
  public function testIsCanonicalUrl(): void {
    // Create a test SEO URL entity.
    $storage = $this->container->get('entity_type.manager')->getStorage('seo_url');
    $seo_url = $storage->create([
      'canonical_url' => ['uri' => 'internal:/test-path'],
      'seo_url' => ['uri' => 'internal:/test-seo-path'],
      'status' => 1,
    ]);
    $seo_url->save();

    // Test with existing SEO URL.
    $this->assertTrue($this->seoUrlManager->isCanonicalUrl('/test-path'));

    // Test with non-existent SEO URL.
    $this->assertFalse($this->seoUrlManager->isSeoUrl('/non-existent-path'));
  }

}
