<?php

declare(strict_types=1);

namespace Drupal\Tests\seo_urls\Unit;

use Drupal\Core\Url;
use Drupal\seo_urls\MenuLinkDestination;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\seo_urls\MenuLinkDestination
 * @group seo_urls
 */
class MenuLinkDestinationTest extends UnitTestCase {

  /**
   * The menu link destination service.
   *
   * @var \Drupal\seo_urls\MenuLinkDestination
   */
  protected $menuLinkDestination;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->menuLinkDestination = new MenuLinkDestination();
  }

  /**
   * Tests trusted callbacks.
   *
   * @covers ::trustedCallbacks
   */
  public function testTrustedCallbacks(): void {
    $this->assertEquals(['preRenderLink'], MenuLinkDestination::trustedCallbacks());
  }

  /**
   * Tests pre-render link with seo_url.create route.
   *
   * @covers ::preRenderLink
   */
  public function testPreRenderLinkWithSeoUrlCreateRoute(): void {
    $url = $this->createMock(Url::class);
    $url->method('isRouted')->willReturn(TRUE);
    $url->method('getRouteName')->willReturn('seo_url.create');

    $element = [
      '#url' => $url,
    ];

    $result = MenuLinkDestination::preRenderLink($element);
    $this->assertArrayHasKey('#cache', $result);
    $this->assertArrayHasKey('contexts', $result['#cache']);
    $this->assertContains('url', $result['#cache']['contexts']);
  }

  /**
   * Tests pre-render link with different route.
   *
   * @covers ::preRenderLink
   */
  public function testPreRenderLinkWithDifferentRoute(): void {
    $url = $this->createMock(Url::class);
    $url->method('isRouted')->willReturn(TRUE);
    $url->method('getRouteName')->willReturn('some.other.route');

    $element = [
      '#url' => $url,
      '#cache' => [],
    ];

    $result = MenuLinkDestination::preRenderLink($element);
    $this->assertArrayNotHasKey('contexts', $result['#cache']);
  }

  /**
   * Tests pre-render link without URL.
   *
   * @covers ::preRenderLink
   */
  public function testPreRenderLinkWithoutUrl(): void {
    $element = [
      '#cache' => [],
    ];

    $result = MenuLinkDestination::preRenderLink($element);
    $this->assertArrayNotHasKey('contexts', $result['#cache']);
  }

}
