<?php

namespace Drupal\sgd_dashboard\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Routing\RedirectDestinationTrait;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Url;
use Drupal\sgd_dashboard\Entity\Bundle\WebsiteBundle;
use Drupal\sgd_dashboard\Exception\SiteGuardianClientAPIException;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Provides route responses for the Website refresh.
 */
class SgdWebsiteRefreshController extends ControllerBase {

  use StringTranslationTrait;
  use RedirectDestinationTrait;

  /**
   * The SGD data service.
   *
   * @var \Drupal\sgd_dashboard\Services\SiteGuardianDataService
   */
  protected $dataService;

  /**
   * The Drupal messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container) {

    $instance = parent::create($container);

    $instance->dataService = $container->get('siteguardian.DataService');
    $instance->messenger = $container->get('messenger');

    return $instance;
  }

  /**
   * Refreshes Website info.
   */
  public function refresh(WebsiteBundle $node, RouteMatchInterface $route_match): RedirectResponse {

    try {

      $this->dataService->refreshWebsiteFields($node);

      $this->messenger->addStatus($this->t("The website %website has been successfully refreshed.", [
        '%website' => $node->getTitle(),
      ]));

    }

    catch (SiteGuardianClientAPIException $e) {

      $message = [];

      $message[] = $this->t("Site Guardian was unable to retrieve status/module information for the Website %website (with url %url).", [
        '%website' => $node->getTitle(),
        '%url' => $node->get('field_url')->value,
      ]);

      $message[] = $this->t("Please check the Site Guardian website settings and manually refresh.");

      $this->messenger->addStatus(implode(' ', $message));
    }

    // Redirect back to where we came from.
    $destination = Url::fromUserInput($this->getRedirectDestination()->get());

    if ($destination->isRouted()) {
      return $this->redirect($destination->getRouteName(), $destination->getRouteParameters());
    }
  }

}
