<?php

declare(strict_types=1);

namespace Drupal\sgd_dashboard\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\link\LinkItemInterface;
use Drupal\sgd_dashboard\SgdEnabledProjectEntityInterface;

/**
 * Defines the site guardian dashboard enabled project entity entity class.
 *
 * @ContentEntityType(
 *   id = "sgd_enabled_project",
 *   label = @Translation("Site Guardian Dashboard Enabled Project entity"),
 *   label_collection = @Translation("Site Guardian Dashboard Enabled Project entities"),
 *   label_singular = @Translation("site guardian dashboard enabled project entity"),
 *   label_plural = @Translation("site guardian dashboard enabled project entities"),
 *   label_count = @PluralTranslation(
 *     singular = "@count site guardian dashboard enabled project entities",
 *     plural = "@count site guardian dashboard enabled project entities",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\sgd_dashboard\SgdEnabledProjectEntityListBuilder",
 *     "views_data" = "Drupal\sgd_dashboard\SgdEnabledProjectEntityViewsData",
 *     "form" = {
 *       "add" = "Drupal\sgd_dashboard\Form\SgdEnabledProjectEntityForm",
 *       "edit" = "Drupal\sgd_dashboard\Form\SgdEnabledProjectEntityForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "delete-multiple-confirm" = "Drupal\Core\Entity\Form\DeleteMultipleForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\sgd_dashboard\Routing\SgdEnabledProjectEntityHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "sgd_enabled_project",
 *   admin_permission = "administer sgd_enabled_project",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "id",
 *     "uuid" = "uuid",
 *   },
 *   links = {
 *     "collection" = "/admin/content/sgd-enabled-project",
 *     "add-form" = "/sgd-enabled-project/add",
 *     "canonical" = "/sgd-enabled-project/{sgd_enabled_project}",
 *     "edit-form" = "/sgd-enabled-project/{sgd_enabled_project}",
 *     "delete-form" = "/sgd-enabled-project/{sgd_enabled_project}/delete",
 *     "delete-multiple-form" = "/admin/content/sgd-enabled-project/delete-multiple",
 *   },
 * )
 */
final class SgdEnabledProjectEntity extends ContentEntityBase implements SgdEnabledProjectEntityInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {

    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Datestamp          field_datestamp            Timestamp.
    $fields['datestamp'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Datestamp'))
      ->setDisplayOptions('view', [
        'type' => 'timestamp',
        'weight' => 0,
      ]);

    // Existing Major      field_existing_major      Text (plain)
    $fields['existing_major'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Existing Major'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Existing Version    field_existing_version    Text (plain)
    $fields['existing_version'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Existing Version'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Includes            field_includes            Text (plain, long)
    $fields['includes'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Includes'))
      ->setRequired(FALSE)
      ->setDisplayOptions('view', [
        'label' => 'visible',
        'type' => 'basic_string',
        'weight' => 1,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => 1,
        'settings' => ['rows' => 4],
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Info                field_info                Text (plain, long)
    $fields['info'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Info'))
      ->setRequired(FALSE)
      ->setDisplayOptions('view', [
        'label' => 'visible',
        'type' => 'basic_string',
        'weight' => 1,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => 1,
        'settings' => ['rows' => 4],
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Install Type        field_install_type        Text (plain)
    $fields['install_type'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Install Type'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Latest Version      field_latest_version      Text (plain)
    $fields['latest_version'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Latest Version'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Link                field_link                Link.
    $fields['link'] = BaseFieldDefinition::create('link')
      ->setLabel(t('Link'))
      ->setSettings([
        'link_type' => LinkItemInterface::LINK_GENERIC,
        'title' => DRUPAL_DISABLED,
      ])
      ->setDisplayOptions('form', [
        'type' => 'link',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Project Status      field_project_status      Text (plain)
    $fields['project_status'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Project Status'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Project Type        field_project_type        Text (plain)
    $fields['project_type'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Project Type'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Recommended        field_recommended          Text (plain)
    $fields['recommended'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Recommended'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Releases            field_releases            Text (plain, long)
    $fields['releases'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Releases'))
      ->setRequired(FALSE)
      ->setDisplayOptions('view', [
        'label' => 'visible',
        'type' => 'basic_string',
        'weight' => 1,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => 1,
        'settings' => ['rows' => 4],
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Security Updates    field_security_updates    Text (plain, long)
    $fields['security_updates'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Security Updates'))
      ->setRequired(FALSE)
      ->setDisplayOptions('view', [
        'label' => 'visible',
        'type' => 'basic_string',
        'weight' => 1,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => 1,
        'settings' => ['rows' => 4],
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Status ID          field_status_id            Number (integer)
    $fields['status_id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Status ID'));

    // Title              field_title                Text (plain)
    $fields['title'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Title'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    // Website  field_website  Entity reference
    // (Reference type: Content Content type: Website)
    $fields['website'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Website'))
      ->setSetting('target_type', 'node')
      ->setSetting('handler', 'default')
      ->setSetting('handler_settings', ['target_bundles' => ['website' => 'website']])
      ->setDisplayOptions('view', [
        'label'  => 'hidden',
        'type'   => 'website',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type'     => 'entity_reference_autocomplete',
        'weight'   => 5,
        'settings' => [
          'match_operator'    => 'CONTAINS',
          'size'              => '60',
          'autocomplete_type' => 'tags',
          'placeholder'       => '',
        ],
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setDescription(t('The time that the site guardian dashboard enabled project entity was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the site guardian dashboard enabled project entity was last edited.'));

    return $fields;
  }

}
