<?php

namespace Drupal\sgd_dashboard\Services;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\sgd_dashboard\Entity\Bundle\WebsiteBundle;
use Drupal\sgd_dashboard\Exception\SiteGuardianClientAPIException;
use GuzzleHttp\Client as httpClient;
use Psr\Log\LoggerInterface;

/**
 * SiteGuardian dashboard API client service.
 *
 * Provides all interaction between the dasboard and the remote
 * websites Site Guardian API instances.
 */
class SiteGuardianAPIClientService {

  use StringTranslationTrait;

  /**
   * The http client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * Logger.
   *
   * @var Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * {@inheritDoc}
   */
  public function __construct(httpClient $httpClient, LoggerInterface $logger) {
    $this->httpClient = $httpClient;
    $this->logger = $logger;
  }

  /**
   * Gets the Status Report for a Website.
   *
   * @return array
   *   Info about Drupal version, PHP version, and more.
   */
  public function getStatus(WebsiteBundle $website): array {

    $this->logger->info("Requesting website status from %website", [
      '%website' => $website->getTitle(),
    ]);

    // Get the URL and if failure throw an error.
    if (!($url = $this->getUrl($website, '/site_guardian/status_report?site_guardian_key='))) {
      throw new SiteGuardianClientAPIException('Invalid URL or Site Guardian key for website ' . $website->getTitle() . '.');
    }

    // Get any basic auth credentials
    $basicAuth = $website->getBasicAuth();

    try {
      if ($basicAuth) {
        $request = $this->httpClient->get($url, ['auth' => [
          $basicAuth['username'], 
          $basicAuth['password'], 
        ]]);
      }

      else {
        $request = $this->httpClient->get($url);
      }
    }

    catch (\Exception $e) {

      $this->logger->error("Getting website status from %website: @error", [
        '%website' => $website->getTitle(),
        '@error' => $e->getMessage(),
      ]);

      throw new SiteGuardianClientAPIException('An unexpected error occured whilst accessing the website ' . $website->getTitle() . '.');
    }

    return json_decode($request->getBody(), TRUE);
  }

  /**
   * Gets a list of the Enabled projects for a Website, along with update info.
   *
   * @return array
   *   List of Enabled projects for a Website, along with update info.
   */
  public function getEnabledProjects(WebsiteBundle $website): array {

    $this->logger->info("Requesting website projects from %website", [
      '%website' => $website->getTitle(),
    ]);

    // Get the URL and if failure throw an error.
    if (!($url = $this->getUrl($website, '/site_guardian/enabled_modules_and_updates?site_guardian_key='))) {
      throw new SiteGuardianClientAPIException('Invalid URL or Site Guardian key for website ' . $website->getTitle() . '.');
    }

    // Get any basic auth credentials
    $basicAuth = $website->getBasicAuth();

    try {
      if ($basicAuth) {
        $request = $this->httpClient->get($url, ['auth' => [
          $basicAuth['username'], 
          $basicAuth['password'], 
        ]]);
      }

      else {
        $request = $this->httpClient->get($url);
      }
    }

    catch (\Exception $e) {

      $this->logger->error("Getting website projects from %website: @error", [
        '%website' => $website->getTitle(),
        '@error' => $e->getMessage(),
      ]);

      throw new SiteGuardianClientAPIException('An unexpected error occurred whilst accessing the website ' . $website->getTitle() . '.');
    }

    return json_decode($request->getBody(), TRUE);
  }

  /**
   * Builds an endpoint URL for the Site Guardian API.
   *
   * @return string
   *   The URL built from the website node.
   */
  private function getUrl(WebsiteBundle $website, $path) : string {

    // Get base URL (without authorisation)
    $baseUrl = $website->getBaseUrl();

    // Get Site Guardian key.
    $siteGuardianKey = !empty($website->get('field_site_guardian_key')->value) ? trim($website->get('field_site_guardian_key')->value, " \t") : '';

    // No URL or Site Guardian Key then return.
    if (empty($baseUrl) || empty($siteGuardianKey)) {
      return FALSE;
    }

    // Build the final URL.
    $path = trim($path, " /\t");

    $url = $baseUrl . '/' . $path . $siteGuardianKey;

    return $url;
  }

}
