<?php

namespace Drupal\sgd_dashboard\Plugin\SgdCompanion;

use Drupal\sgd_dashboard\SgdCompanionPluginBase;

/**
 * Provides a SGD Companion plugin.
 *
 * @SgdCompanion(
 *   id = "sgd_companion_user_status",
 * )
 */
class SgdCompanionUserStatus extends SgdCompanionPluginBase {

  /**
   * {@inheritdoc}
   */
  public function canProcessStatus($statusData) : bool {

    // If the Site Guardian API Companion module 'User status' is installed on
    // the site we should have data from it.
    // Just checking for the existance of 'sgd_user_roles' is enough.
    if (array_key_exists('sgd_user_roles', $statusData)) {
      return TRUE;
    }
    else {
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function saveStatus($websiteData, $statusData, $enabledProjects = NULL) : bool {

    if ($this->canProcessStatus($statusData)) {

      $userData = [
        'sgd_user_count'    => $statusData['sgd_user_count'],
        'sgd_user_status'   => $statusData['sgd_user_status'],
        'sgd_user_1_status' => $statusData['sgd_user_1_status'],
        'sgd_user_activity' => $statusData['sgd_user_activity'],
        'sgd_user_creation' => $statusData['sgd_user_creation'],
        'sgd_user_roles'    => $statusData['sgd_user_roles'],
      ];

      // Admin count may not exist if the administrator account has been
      // deleted.
      if (array_key_exists('sgd_user_admin_count', $statusData)) {
        $userData['sgd_user_admin_count'] = $statusData['sgd_user_admin_count'];
      }
      else {
        $userData['sgd_user_admin_count'] = [
          'title' => "Number of users with the 'administrator' role.",
          'value' => "Role not found.",
          'description' => "Administrator (administrator) role was not found.",
          'severity' => 0,
        ];
      }

      $websiteData->set('data_user_status', serialize($userData));

    }
    else {
      $websiteData->set('data_user_status', NULL);
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatusDefaults() : array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus($websiteData) : array | NULL {

    if ($dataSerialized = $websiteData->get('data_user_status')->value) {
      $data = unserialize($dataSerialized, ['allowed_classes' => FALSE]);
    }
    else {
      $data = NULL;
    }

    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function getBuildElements($websiteData) : array | NULL {

    if ($data = $this->getStatus($websiteData)) {

      $elements = [];

      $validation = $this->validate($data);

      foreach ($data as $key => $value) {

        if ($key != 'sgd_user_roles') {
          $elements[$key] = [
            'title' => $value['title'],
            'value' => $value['value'],
            'status' => $validation[$key] ?? NULL,
            'description' => $value['description'],
          ];
        }
      }

      foreach ($data['sgd_user_roles'] as $key => $value) {
        $elements['sgd_user_roles'][$key] = [
          'title' => $value['label'],
          'value' => $value['count'],
        ];
      }

      return $elements;
    }

    return NULL;
  }

  /**
   * Validate data.
   *
   * Checks each item and returns a good/nuetral/bad status that can be
   * displayed on page or in a report.
   */
  private function validate(&$statusData): array {

    $validation = [];

    // Validate User variables.
    // Each validation is hard coded as it differs for each plugin.
    foreach ($statusData as $key => $status) {

      switch ($key) {

        case 'sgd_user_1_status':

          if ($status['value'] == 'Enabled') {
            $validation[$key] = [
              'class' => 'warning',
              'text' => $this->t('Alert'),
              'message' => $this->t('Consider disabling the admin account to reduce the potential for brute force password attacks.'),
            ];
          }
          else {
            $validation[$key] = [
              'class' => 'ok',
              'text' => $this->t('OK'),
            ];
          }

          break;

        case 'sgd_user_activity':

          [,, $neverUsed] = explode('/', $status['value']);

          if ($neverUsed > 0) {
            $validation[$key] = [
              'class' => 'warning',
              'text' => $this->t('Alert'),
              'message' => $this->t('Consider disabling or removing accounts that have never been used.'),
            ];
          }
          break;

        case 'sgd_user_admin_count':
          
          if ($status['value'] == 'Role not found.') {
            $validation[$key] = [
              'class' => 'warning',
              'text' => $this->t('Alert'),
              'message' => $this->t('Set the role used for the websites site administration in the sgd_user module settings.'),
            ];
          }

          break;
      }

    }

    return $validation;
  }

}
