<?php

namespace Drupal\sgd_dashboard\Controller;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\node\Controller\NodeViewController;
use Drupal\views\Views;

use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides route responses for the PHP data page.
 */
class SgdWebsiteViewController extends NodeViewController {

  use StringTranslationTrait;

  /**
   * The SGD data service.
   *
   * @var \Drupal\sgd_dashboard\Services\SiteGuardianDataService
   */
  protected $dataService;

  /**
   * The SGD cron service.
   *
   * @var \Drupal\sgd_dashboard\Services\SiteGuardianCronService
   */
  protected $cronService;

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container) {

    $instance = parent::create($container);

    $instance->dataService = $container->get('siteguardian.DataService');
    $instance->cronService = $container->get('siteguardian.CronService');

    return $instance;
  }

  /**
   * Display the PHP data for the website.
   *
   * @return array
   *   A render array as expected by
   *   \Drupal\Core\Render\RendererInterface::render().
   */
  public function view(EntityInterface $node, $view_mode = 'full', $langcode = NULL) {

    // Only want to affect website bundles so if anything else just call the
    // parent and return.
    /** @var \Drupal\sgd_dashboard\Entity\Bundle\WebsiteBundle $node */
    if ($node->bundle() !== 'website') {
      return parent::view($node, $view_mode, $langcode);
    }

    // Get the crontab for the website.
    $crontab = $this->cronService->getCronTabAsText($node);

    // Add the SG config information for the site being viewed.
    $build['#status']['website'] = [
      'client_name' => [
        'title' => $this->t('Client'),
        'value' => $node->field_client->entity->getTitle(),
      ],
      'url' => [
        'title' => $this->t('URL'),
        'value' => Url::fromUri($node->field_url->value),
      ],
      'crontab' => [
        'title' => $this->t('Refresh crontab schedule'),
        'value' => $crontab,
      ],
      'api_key_unique' => [
        'title' => $this->t('Site Guardian API key unique'),
        'value' => $this->dataService->isSiteGuardianKeyUnique($node->field_site_guardian_key->value) ? $this->t('Yes') : $this->t('No'),
      ],
      'last_checked' => [
        'title' => $this->t('Last checked'),
        'value' => $node->field_site_last_checked->value ?? 'Never',
      ],
      'visit_site_link' => [
        'title' => $this->t('Visit site'),
        'value' => [
          '#type' => 'link',
          '#title' => $this->t('Visit site'),
          '#url' => Url::fromRoute('entity.node.visit_website', [
            'node' => $node->id(),
          ]),
          '#attributes' => [
            'class' => ['button'],
            'target' => ['_blank'],
          ],
        ],
      ],
    ];

    // Add a 'Refresh' operation if user has permission.
    if (\Drupal::currentUser()->hasPermission('refresh websites')) {

      $build['#status']['website']['refresh_link'] = [
        'title' => $this->t('Refresh'),
        'value' => [
          '#type' => 'link',
          '#title' => $this->t('Refresh'),
          '#url' => Url::fromRoute('entity.node.refresh_website',
            [
              'node' => $node->id(),
            ],
            [
              'query' => [
                'destination' => Url::fromRoute('entity.node.canonical', ['node' => $node->id()])->toString(),
              ],
            ],
          ),
          '#attributes' => [
            'class' => ['button'],
          ],
        ],
      ];
    }

    // Add a 'Download report' button if the PDF report download route exists
    // and user has permission and the site has been scanned.
    /** @var \Drupal\Core\Routing\RouteProviderInterface $route_provider */
    $route_provider = \Drupal::service('router.route_provider');

    if (count($route_provider->getRoutesByNames(['sgd_pdf_report.download_report'])) === 1) {

      if (\Drupal::currentUser()->hasPermission('download site guardian report')) {

        if (!empty($node->field_site_last_checked->value)) {

          $build['#status']['website']['download_report_link'] = [
            'title' => $this->t('Download report'),
            'value' => [
              '#type' => 'link',
              '#title' => $this->t('Download report'),
              '#url' => Url::fromRoute('sgd_pdf_report.download_report',
                [
                  'node' => $node->id(),
                ],
              ),
              '#attributes' => [
                'class' => ['button'],
              ],
            ],
          ];
        }
      }
    }

    // Get the website data for the node.
    /** @var \Drupal\sgd_dashboard\Entity\SgdWebsiteData $websiteData */
    $websiteData = $node->getDataEntity();

    // Get the plugin companion manager.
    $pluginManager = \Drupal::service('plugin.manager.sgd_companion');

    // Get the core companion plugin.
    $coreStatus = $pluginManager->createInstance('sgd_companion_sgd_core', []);

    // Add the core data module data.
    if ($data = $coreStatus->getBuildElements($websiteData)) {
      $build['#status'] += $data;
    }

    // Get the php companion plugin.
    $phpStatus = $pluginManager->createInstance('sgd_companion_php_status', []);

    // If the site has the PHP Companion module then store the data under 'php'.
    if ($data = $phpStatus->getBuildElements($websiteData)) {
      $build['#status']['php'] = $data;
    }

    // Now build the "projects by website and name" exposed filter.
    // This is from the websites projects view.
    $view = Views::getView('website_projects');

    $view->setDisplay('projects_by_website_and_name');

    $view->initHandlers();

    /** @var \Drupal\views\Plugin\views\exposed_form\ExposedFormPluginInterface $exposedForm */
    $exposedForm = $view->display_handler->getPlugin('exposed_form');

    $build['#exposed_form'] = $exposedForm->renderExposedForm(TRUE);

    // Now the website projects list.
    $build['#project_list'] = [
      '#title' => 'Website projects',
      'view' => [
        '#type' => 'view',
        '#name' => 'website_projects',
        '#display_id' => 'projects_by_website_and_name',
        '#embed' => TRUE,
      ],
    ];

    $build['#theme'] = 'sgd_core_data';

    $build['#attached']['library'][] = 'sgd_dashboard/sgd_core_data';
    $build['#attached']['library'][] = 'sgd_dashboard/sgd_views';

    return $build;
  }

}
