<?php

namespace Drupal\sgd_import_export\Form;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\sgd_dashboard\Services\SiteGuardianCronService;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Provides a Website export form.
 */
class ExportWebsitesForm extends FormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected readonly Connection $database;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected readonly EntityTypeManagerInterface $entityTypeManager;

  /**
   * The Site Guardian cron service.
   *
   * @var \Drupal\sgd_dashboard\Services\SiteGuardianCronService
   */
  protected readonly SiteGuardianCronService $cronService;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $account;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {

    $instance = parent::create($container);

    $instance->database = $container->get('database');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->cronService = $container->get('siteguardian.CronService');
    $instance->account = $container->get('current_user');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'website_export_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    // Get all website entities.
    $websites = $this->entityTypeManager->getStorage('node')->loadByProperties(['type' => 'website']);

    // Build list of options.
    $options = [];

    /** @var \Drupal\sgd_dashboard\Entity\Bundle\WebsiteBundle $website */
    foreach ($websites as $nodeId => $website) {
      $options[$nodeId] = $website->getTitle();
    }

    // Build the form.
    $form['websites'] = [
      '#type' => 'select',
      '#title' => $this->t('Websites'),
      '#description' => $this->t('Select the websites to export.'),
      '#options' => $options,
      '#required' => TRUE,
      '#size' => 10,
      '#multiple' => TRUE,
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Export'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {

    // Get selected websites from the form.
    $selectedWebsites = $form_state->getValue('websites');

    $websites = $this->entityTypeManager->getStorage('node')->loadMultiple($selectedWebsites);

    /** @var \Drupal\sgd_dashboard\Entity\Bundle\WebsiteBundle $website */
    foreach ($websites as $id => $website) {

      $id = $website->getFileNameFromTitle();

      // Get the crontab for the website.
      $crontab = $this->cronService->getCronTab($website);

      // Build the data to export.
      $data[$id] = [
        'name' => trim($website->title->value),
        'client' => trim($website->field_client->entity->getTitle()),
        'environment_type' => !empty($website->field_environment_type->entity) ? $website->field_environment_type->entity->name->value : '',
        'title_suffix' => !empty($website->field_title_suffix->value) ? trim($website->field_title_suffix->value) : '',
        'url' => trim($website->field_url->value),
        'crontab' => !empty($crontab) ? $crontab : '',
      ];

      // If user has permission then add any basic auth and site guardian key.
      if ($website->access('update', $this->account)) {
        $data[$id]['http_auth_user'] = !empty($website->field_http_auth_user->value) ? trim($website->field_http_auth_user->value) : '';
        $data[$id]['http_auth_password'] = !empty($website->field_http_auth_password->value) ? trim($website->field_http_auth_password->value) : '';
        $data[$id]['site_guardian_key'] = trim($website->field_site_guardian_key->value);
      }
    }

    // Create a sensible filename.
    $filename = date('Ymd-His') . '-sgd-website-export.yml';

    // Get the data as a 'yaml' string.
    $yml = YAML::encode($data);

    // // Create the response
    $response = new Response($yml, 200);

    $response->headers->set('Content-Type', 'text/csv; charset=UTF-8');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename);
    $response->headers->set('Content-Transfer-Encoding', 'binary');
    $response->headers->set('Content-Length', (string) strlen($response->getContent()));
    $response->headers->set('Content-Description', 'File Transfer');

    $form_state->setResponse($response);
  }

}
